package pickewanalysis;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Composite;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import pickewanalysis.ewmessages.CodaSCNL;
import pickewanalysis.ewmessages.EWMessage;
import pickewanalysis.ewmessages.EWMessageGroup;
import pickewanalysis.ewmessages.PickSCNL;
import pickewanalysis.ewpicker.PickEWDebugMsg;
import pickewanalysis.ewpicker.PickEWStation;
import pickewanalysis.waveform.WaveForm;
import pickewanalysis.waveform.WaveFormGroup;

public class TracePanel extends javax.swing.JPanel {

    public TracePanel() {
        initComponents();
        TracePanelMouseListener listener = new TracePanelMouseListener(this);
        addMouseListener(listener);
        addMouseMotionListener(listener);
    }

    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 400, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 300, Short.MAX_VALUE)
        );
    }// </editor-fold>//GEN-END:initComponents
    // Variables declaration - do not modify//GEN-BEGIN:variables
    // End of variables declaration//GEN-END:variables

    public void setActiveWaveforms(WaveFormGroup waveGroup,
            WaveFormGroup rdatGroup,
            WaveFormGroup charGroup,
            WaveFormGroup ltaGroup,
            WaveFormGroup staGroup,
            WaveFormGroup staltaGroup) {
        groups[0] = waveGroup;
        groups[1] = rdatGroup;
        groups[2] = charGroup;
        groups[3] = ltaGroup;
        groups[4] = staGroup;
        groups[5] = staltaGroup;
    }

    @Override
    public void paintComponent(Graphics g) {
        super.paintComponent(g);

        //Check if dimensions have been changed
        if (traceImage == null
                || this.getWidth() != traceImage.getWidth()
                || this.getHeight() != traceImage.getHeight()
                || requiresUpdate) {
            //Produce image for panel
            traceImage = produceTraceImage();
        }
        if (traceImage != null) {
            g.drawImage(traceImage, 0, 0, null);
        }
        //Indicate that this is updated
        requiresUpdate = false;

        //Check for drag and drop
        ((TracePanelMouseListener) this.getMouseListeners()[0]).drawBox((Graphics2D) g);
    }

    private BufferedImage produceTraceImage() {

        //Create output image
        BufferedImage image = new BufferedImage(this.getWidth(), this.getHeight(), BufferedImage.TYPE_INT_RGB);
        Graphics2D g = image.createGraphics();
        //Clear image
        g.setColor(Color.WHITE);
        g.fillRect(0, 0, this.getWidth(), this.getHeight());
        int xmin = 0;
        int xmax = getWidth();
        //Set Font
        g.setFont(new Font("Tahoma", Font.PLAIN, 10));
        //Check if there is waveform data to print
        if (groups != null) {
            //Prepare list of visible picks
            visiblePicks = new ArrayList<VisiblePick>();
            //Plot raw data trace
            int ymin = 0;
            int ymax = (int) ((double) getHeight() * this.RAW_HEIGHT_RATIO);
            plotTrace(g, groups[0], xmin, xmax, ymin,
                    ymax, Color.BLACK, true);

            //Plot characteristic function trace
            ymin = ymax;
            ymax = ymin + (int) ((double) getHeight() * this.RDAT_HEIGHT_RATIO);
            plotTrace(g, groups[2], xmin, xmax, ymin,
                    ymax, Color.LIGHT_GRAY, false);
            //Plot filtered function trace
            double[] plotParams = plotTrace(g, groups[1], xmin, xmax, ymin,
                    ymax, Color.BLUE, true);
            //Plot horizontal lines for the MinPeakSize parameter
            int[] mpsPos = getPos(new double[]{0, station.MinPeakSize, 0, -station.MinPeakSize}, plotParams);
            g.setColor(Color.RED);
            g.drawLine(xmin, (mpsPos[1]>ymin)?mpsPos[1]:ymin, xmax, (mpsPos[1]>ymin)?mpsPos[1]:ymin);
            g.drawLine(xmin, (mpsPos[3]<ymax)?mpsPos[3]:ymax, xmax, (mpsPos[3]<ymax)?mpsPos[3]:ymax);
            //Plot detected picks
            PickSCNL last = null;
            for (EWMessage ewm : msg) {
                if (ewm.getType() <= 0) {
                    //This is a debug message, discard
                    continue;
                }
                //Scrutinate picks or codas
                if (ewm.getType() == EWMessage.TYPE_PICK_SCNL) {
                    PickSCNL pick = (PickSCNL) ewm;
                    //Draw Vertical Line
                    int[] pickPos = getPos(new double[]{pick.arrivalTime, 0}, plotParams);
                    g.setColor(Color.RED);
                    g.drawLine(pickPos[0], ymin, pickPos[0], ymax);

                    //Draw First Motion Text
                    char motion = (pick.FirstMotion != ' ') ? pick.FirstMotion : '?';
                    String fmt = String.format(Locale.UK, "%c%1d", motion, pick.Weight);
                    g.drawString(fmt, pickPos[0] + 1, ymin + 12);

                    //Add to list of visible picks
                    addToVisiblePicks(new VisiblePick(pick, new int[]{pickPos[0], ymin, ymax}));

                    //Set this pick as the last, waiting for a coda
                    last = pick;
                } else if (ewm.getType() == EWMessage.TYPE_CODA_SCNL) {
                    CodaSCNL coda = (CodaSCNL) ewm;
                    //Check if there is a last pick
                    if (last == null) {
                        //There is no reference pick... quit
                        continue;
                    }
                    //Draw Vertical Line
                    double codaTime = last.arrivalTime + Math.abs(coda.duration);
                    int[] codaPos = getPos(new double[]{codaTime, 0}, plotParams);
                    g.setColor(Color.GREEN);
                    g.drawLine(codaPos[0], ymin, codaPos[0], ymax);

                    //Add to list of visible picks
                    addToVisiblePicks(new VisiblePick(coda, new int[]{codaPos[0], ymin, ymax}));

                    //Draw transparent square
                    int[] pickPos = getPos(new double[]{last.arrivalTime, 0}, plotParams);
                    Rectangle rect = new Rectangle(pickPos[0] + 1, ymin,
                            codaPos[0] - pickPos[0] - 1, (ymax - ymin - 1));
                    Composite originalComposite = g.getComposite();
                    g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.2f));
                    g.setPaint(Color.GREEN);
                    g.fill(rect);
                    g.setComposite(originalComposite);

                    //Draw duration text
                    String fmt = String.format(Locale.UK, "%3ds", coda.duration);
                    g.setColor(new Color(0, 128, 0)); //Dark Green
                    g.drawString(fmt, codaPos[0] + 1, ymin + 12);

                    //Clear last pick
                    last = null;
                }
            }

            //Plot STA
            ymin = ymax;
            ymax = ymin + (int) ((double) getHeight() * this.STA_HEIGHT_RATIO);
            plotTrace(g, groups[3], xmin, xmax, ymin,
                    ymax, Color.LIGHT_GRAY, false);
            //Plot LTA
            plotTrace(g, groups[4], xmin, xmax, ymin,
                    ymax, Color.DARK_GRAY, false);

            //Plot STA/LTA
            ymin = ymax;
            ymax = ymin + (int) ((double) getHeight() * this.THRES_HEIGHT_RATIO);
            plotParams = plotTrace(g, groups[5], xmin, xmax, ymin,
                    ymax, Color.LIGHT_GRAY, false);
            //Plot Threshold line
            int[] thPos = getPos(new double[]{0, station.EventThresh}, plotParams);
            g.setColor(Color.RED);
            g.drawLine(xmin, (thPos[1]>ymin)?thPos[1]:ymin, xmax, (thPos[1]>ymin)?thPos[1]:ymin);
            //Plot false alarms
            for (EWMessage ewm : msg) {
                if (ewm.getType() >= 0) {
                    continue; //These are real picks
                }
                PickEWDebugMsg fp = (PickEWDebugMsg) ewm;
                int[] pos = getPos(new double[]{(double) fp.getTime() / 1000.0, 0}, plotParams);
                g.setColor(new Color(128, 0, 0)); //Dark Red
                g.drawLine(pos[0], ymin, pos[0], thPos[1]);

                //Add to list of visible picks
                addToVisiblePicks(new VisiblePick(fp, new int[]{pos[0], ymin, thPos[1]}));

                String fmt = String.format(Locale.UK, "%d", -fp.getType());
                g.drawString(fmt, pos[0] + 1, ymin + 12);
            }
        }
        //Draw squares around plots, regardless of having data or not
        int ymin = 0;
        int ymax = (int) ((double) getHeight() * this.RAW_HEIGHT_RATIO);
        g.setColor(Color.BLACK);
        g.drawRect(xmin, ymin, (xmax - xmin - 1), (ymax - ymin));
        g.setColor(Color.BLACK);
        g.drawString("Raw Trace", xmin + 2, ymin + 10);

        ymin = ymax;
        ymax = ymin + (int) ((double) getHeight() * this.RDAT_HEIGHT_RATIO);
        g.setColor(Color.BLACK);
        g.drawRect(xmin, ymin, (xmax - xmin - 1), (ymax - ymin));
        g.setColor(Color.BLUE);
        g.drawString("High-Pass Filtered Trace", xmin + 2, ymin + 10);
        g.setColor(Color.LIGHT_GRAY);
        g.drawString("Characteristic Function", xmin + 2, ymin + 20);
        g.setColor(Color.RED);
        g.drawString("MinPeakSize", xmin + 2, ymin + 30);

        ymin = ymax;
        ymax = ymin + (int) ((double) getHeight() * this.STA_HEIGHT_RATIO);
        g.setColor(Color.BLACK);
        g.drawRect(xmin, ymin, (xmax - xmin - 1), (ymax - ymin));
        g.setColor(Color.BLACK);
        g.drawString("Long-Term Average", xmin + 2, ymin + 10);
        g.setColor(Color.LIGHT_GRAY);
        g.drawString("Short-Term Average", xmin + 2, ymin + 20);

        ymin = ymax;
        ymax = ymin + (int) ((double) getHeight() * this.THRES_HEIGHT_RATIO);
        g.setColor(Color.BLACK);
        g.drawRect(xmin, ymin, (xmax - xmin - 1), (ymax - ymin));
        g.setColor(Color.LIGHT_GRAY);
        g.drawString("Threshold Function", xmin + 2, ymin + 10);
        g.setColor(Color.RED);
        g.drawString("Threshold", xmin + 2, ymin + 20);
        return image;
    }

    //Method to compute a screen xy position with plot parameters
    private int[] getPos(double[] p, double[] plotParams) {
        int np = p.length;
        int[] output = new int[p.length];
        for (int i = 0; i < np; i += 2) {
            output[i] = (int) plotParams[0] + (int) ((p[i] - plotParams[1]) * plotParams[2] + 0.5);
            output[i + 1] = (int) plotParams[3] - (int) ((p[i + 1] - plotParams[4]) * plotParams[5]);
        }
        return output;
    }

    //Method to plot a trace within a given set of limits
    //Returns a set of scaling parameters for other functions
    private double[] plotTrace(Graphics2D gr, WaveFormGroup group, int xmin, int xmax, int ymin,
            int ymax, Color color, boolean centerVerticaly) {
        //Find out limits of the data
        double yAverage = 0;
        double yMax = Double.NEGATIVE_INFINITY;
        double yMin = Double.POSITIVE_INFINITY;
        int nsamp = 0;
        for (WaveForm wave : group) {
            double[] samples = wave.getSamples();
            for (int i = 0; i < samples.length; i++) {
                // Compute sample instant
                long t = (long) (wave.getStartTime() + (double) i * 1000.0 / wave.getSamplingRate());
                //Check if is within the limits
                if (t >= starttime && t <= endtime) {
                    yAverage += samples[i];
                    nsamp++;
                }
            }
            //nsamp += wave.getSampleCount();
        }
        yAverage /= (double) nsamp;

        double yAvg = (centerVerticaly) ? yAverage : 0;
        int yZero = (centerVerticaly) ? ((int) ((ymax + ymin) / 2 + 0.5)) : ymax;

        //Calculate data limits
        for (WaveForm wave : group) {
            double[] samples = wave.getSamples();
            for (int i = 0; i < samples.length; i++) {
                //Compute sample instant
                long t = (long) (wave.getStartTime() + (double) i * 1000.0 / wave.getSamplingRate());
                //Check if is within the limits
                if (t >= starttime && t <= endtime) {
                    if ((samples[i] - yAvg) > yMax) {
                        yMax = samples[i];
                    }
                    if ((samples[i] - yAvg) < yMin) {
                        yMin = samples[i];
                    }
                }
            }
        }
        //Define vertical scaling limits
        double yScale = (centerVerticaly)
                ? ((double) (ymax - ymin) / 2
                / Math.max(Math.abs(yMax), Math.abs(yMin)))
                : ((double) (ymax - ymin) / Math.abs(yMax));

        //Define horizontal scaling limits
        double xScale = (double) (xmax - xmin) / (double) (endtime - starttime) * 1000.0;
        int xZero = xmin;

        //Start plotting data
        gr.setColor(color);
        int lastX = 0, lastY = 0;
        boolean firstwave = true;
        for (WaveForm wave : group) {
            //For each wave, plot the samples in the correct horizontal position
            double samprate = wave.getSamplingRate();
            double[] samples = wave.getSamples();
            int[] xPos = new int[wave.getSampleCount()];
            int[] yPos = new int[wave.getSampleCount()];
            for (int i = 0; i < wave.getSampleCount(); i++) {
                xPos[i] = xZero + (int) (((double) (wave.getStartTime() - starttime) / 1000.0
                        + (double) i / samprate) * xScale);
                yPos[i] = yZero - (int) ((samples[i] - yAvg) * yScale);
            }

            //Plot the samples in this wave
            gr.drawPolyline(xPos, yPos, xPos.length);

            //Link with previous wave
            if (!firstwave) {
                gr.drawLine(lastX, lastY, xPos[0], yPos[0]);
            } else {
                firstwave = false;
            }

            //Save the last position to link consecutive waves
            lastX = xPos[wave.getSampleCount() - 1];
            lastY = yPos[wave.getSampleCount() - 1];
        }

        return new double[]{(double) xZero, (double) starttime / 1000.0, xScale,
                    (double) yZero, yAvg, yScale};
    }

    private void addToVisiblePicks(VisiblePick x) {
        if (x.pos[0] < 0 || x.pos[0] > getWidth()) {
            return;
        }
        visiblePicks.add(x);
    }

    public String isPick(int x, int y) {
        if (visiblePicks == null) {
            return null;
        }
        int d = Integer.MAX_VALUE;
        String output = null;
        for (VisiblePick pick : visiblePicks) {
            if (y < pick.pos[1] || y > pick.pos[2]) {
                continue;
            }
            int newd = (int) Math.abs(pick.pos[0] - x);
            if (newd < d && newd < 5) {
                //Fuond a better candidate;
                d = newd;
                switch (pick.ewm.getType()) {
                    case EWMessage.TYPE_PICK_SCNL:
                        output = ((PickSCNL) pick.ewm).toString();
                        break;
                    case EWMessage.TYPE_CODA_SCNL:
                        output = ((CodaSCNL) pick.ewm).toString();
                        break;
                    default:
                        output = ((PickEWDebugMsg) pick.ewm).toString();
                        break;
                }
            }
        }
        return output;
    }

    public void update(long starttime, long endtime, PickEWStation station,
            WaveFormGroup[] groups, EWMessageGroup mg) {
        this.starttime = starttime;
        this.endtime = endtime;
        this.groups = groups;
        this.msg = mg;
        this.station = station;
        //Set to update
        requiresUpdate = true;
        repaint();
    }
    //Method to return the position of any sample on the image
    //Custom variables
    BufferedImage traceImage = null;
    boolean requiresUpdate = true;
    private List<VisiblePick> visiblePicks;
    //Information to be regularly updated
    public long starttime, endtime;
    private WaveFormGroup[] groups;
    private EWMessageGroup msg;
    private PickEWStation station;
    //Constants
    private final double RAW_HEIGHT_RATIO = 0.15;
    private final double RDAT_HEIGHT_RATIO = 0.45;
    private final double STA_HEIGHT_RATIO = 0.15;
    private final double THRES_HEIGHT_RATIO = 0.25;
}

class VisiblePick {

    EWMessage ewm;
    int[] pos;

    VisiblePick(EWMessage ewm, int[] pos) {
        this.ewm = ewm;
        this.pos = pos;
    }
}