/*
*
*    Based on SUDS putaway and Geomag WDC putaway.
*    For Erol Kalcan
*
*/

/* cosmos0putaway.c

Routines for writing trace data in COSMOS V0 format.

Volume 0 files have raw data values in digital counts, obtained directly from the native binary files of
the recording instrument. (Records obtained by digitizing analog film generally do not have a V0
file.) V0 files may have almost no quality control or checking, and so they may be treated as internal
files by networks. V0 files have adequate information in the headers to be converted to files with
physical values. There is one component per file, so one record obtained by a triaxial recorder will
result in three files.

Header information is populated by a database of files. We don't need to know
about it after we use the SCNL to find the right file and prepend it.

Count data is extracted from a wave server:

Data Section:
 The first line of the data section includes the type of data (accel., velocity, displ., etc.) and its
units, the number of data points, and the format to be used in reading the data. The format
will type be for integer or real values, as appropriate; the format for real values can be
floating point (e.g., 8f10.5) or exponential (e.g., 5e16.7). Although not required, 80-character
line lengths are most convenient for many engineering strong motion data users.
 The first line is followed by as many lines of data as are needed to accommodate the number
of points, given the format in which the data values are written.

In our case we'll alway use units=counts, and interger Format=(10I8):
First line example followed by a data line.
17629    raw accel.   pts, approx  176 secs, units=counts (50),Format=(10I8)
-28596  -28611  -28630  -28617  -28609  -28550  -28543  -28654  -28698  -28661

*/

#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "earthworm.h"
#include "trace_buf.h"
#include "swap.h"
#include "ws_clientII.h"
#include "cosmos0head.h"
#include "cosmos0putaway.h"
#include "pa_subs.h"
#include "earthworm_simple_funcs.h"
#include "chron3.h"

#define TAG_FILE        '.tag'        /* file containing the last known file tag */
#define MAXTXT           150

FILE    *COSMOS0fp;                      /* file pointer for the COSMOS0 file */
FILE    *COSMOS0Libfp;                      /* file pointer for the COSMOS0 library file */

static  long   *COSMOS0Buffer;           /* write out COSMOS0 data as long integers */
static  char    COSMOS0OutputFormat[MAXTXT];
static  long    LineMean;            /* average of the 60 samples in the line */

/* Internal Function Prototypes */
static int StructMakeLocal(void *, int, char, int);

/************************************************************************
* Initialization function,                                              *
*       This is the Put Away startup intializer. This is called when    *
*       the system first comes up. Here is a chance to look around      *
*       and see if it's possible to do business, and to complain        *
*       if not ,BEFORE an event has to be processed.                    *
*                                                                       *
*       For PCCOSMOS0, all we want to do is to make sure that the       *
*       directory where files should be written exists.                 *
*************************************************************************/
int COSMOS0PA_init(int OutBufferLen, char *OutDir, char *OutputFormat,
	int debug)
{
	/** Allocate COSMOS0Buffer and COSMOS0BufferShort
	We waste RAM by allocating both the long and short buffers here
	at the beginning of the code because some fluke (feature?) of NT
	which I don't understand becomes unhappy if we do the allocation
	later. Win2000, of course, behaves differently, and is quite happy
	with buffer allocation after we have determined the format of the
	incoming data */
	if ((COSMOS0Buffer = (long *)malloc(OutBufferLen * sizeof(char))) == NULL)
	{
		logit("et", "COSMOS0PA_init: couldn't malloc COSMOS0Buffer\n");
		return EW_FAILURE;
	}

	/* Make sure that the top level output directory exists */
	if (RecursiveCreateDir(OutDir) != EW_SUCCESS)
	{
		logit("e", "COSMOS0PA_init: Call to RecursiveCreateDir failed\n");
		return EW_FAILURE;
	}

	if (strlen(OutputFormat) >= sizeof(COSMOS0OutputFormat))
	{
		logit("", "COSMOS0PA_init: Error: OutputFormat(%s) is too long! Quitting!\n",
			OutputFormat);
		return(EW_FAILURE);
	}
	else
	{
		strcpy(COSMOS0OutputFormat, OutputFormat);
	}
	return EW_SUCCESS;
}

/************************************************************************
*   This is the Put Away event initializer. It's called when a snippet  *
*   has been received, and is about to be processed.                    *
*   It gets to see the pointer to the TraceRequest array,               *
*   and the number of loaded trace structures.                          *
*                                                                       *
*   We need to make sure that the target directory                      *
*   exists, create it if it does not, then open the COSMOS0 file        *
*   for writing.                                                        *
*                                                                       *
*   This also opens the library file, and writes its contents to our    *
*   output file                                                         *
*************************************************************************/
int COSMOS0PA_next_ev(TRACE_REQ *ptrReq,
	char *OutDir, char *LibDir, char *EventDate, char *EventTime,
	int cadence, int debug, double Multiplier)

{
	char    COSMOS0File[4 * MAXTXT];
	char    COSMOS0LibFile[4 * MAXTXT];
/*	char    c; */
	char    year[5];
	char    yr[3];
	char    mo[3];
	char    dy[3];
	char    hr[3];
	char    mn[3];
	char    sec[7];
	char	cos_date[35];
	char    record_id[35];
	char    str[COSMOSLINELEN];
	char    tempstr[COSMOSLINELEN];

	int     LineNumber = 0;

	/* Grab the date-related substrings that we need for filenames. */
	strncpy(year, EventDate, 4);
	year[4] = '\0';
	strncpy(yr, EventDate + 2, 2);
	yr[2] = '\0';
	strncpy(mo, EventDate + 4, 2);
	mo[2] = '\0';
	strncpy(dy, EventDate + 6, 2);
	dy[2] = '\0';

	strncpy(hr, EventTime, 2);
	hr[2] = '\0';
	strncpy(mn, EventTime + 2, 2);
	mn[2] = '\0';
	strncpy(sec, EventTime + 4, 5);
	sec[5] = '\0';

	cos_date[34] = '\0';
	record_id[34] = '\0';	
	tempstr[0]='\0';


	sprintf(COSMOS0File, "%s/%s%s%s%s.v0", OutDir,
		ptrReq->sta, yr, mo, dy);

	/* cos_date "06/17/2018, 18:34:38.004 GMT (Q=5) "<-35 chars; Q stands for Quality, we have no way to know that */
	sprintf(cos_date, "%s/%s/%s, %s:%s:%s  GMT", mo, dy, year, hr, mn, sec);

	RecursiveCreateDir(OutDir);
	sprintf(COSMOS0LibFile, "%s/%s_%s_%s_%s.dlv0", 
		LibDir, ptrReq->net, ptrReq->sta, ptrReq->loc, ptrReq->chan);

	if (debug == 1)
		logit("t", "Opening COSMOS0 library file header %s\n", COSMOS0LibFile);

	/* open library file just for reading */
	if ((COSMOS0Libfp = fopen(COSMOS0LibFile, "r")) == NULL)
	{
		logit("e", "COSMOS0PA_next_ev: unable to open file %s: %s\n",
			COSMOS0LibFile, strerror(errno));
		return EW_FAILURE;
	}

	if (debug == 1)
		logit("et", "Opening COSMOS0 file %s\n", COSMOS0File);

	/* open file */
	if ((COSMOS0fp = fopen(COSMOS0File, "wt")) == NULL)
	{
		logit("e", "COSMOS0PA_next_ev: unable to open file %s: %s\n",
			COSMOS0File, strerror(errno));
		return EW_FAILURE;
	}


	/* Copy the library file to the putaway file*/
/*	c = fgetc(COSMOS0Libfp);
	while (c != EOF)
	{
		fputc(c, COSMOS0fp);
		c = fgetc(COSMOS0Libfp);
	}
*/
	while (fgets(str, COSMOSLINELEN, COSMOS0Libfp) != NULL) {
		LineNumber++ ;
		switch (LineNumber) {
		case 2: /*  2 1-40 Earthquake name (before a name has been assigned, may appear as Record of; test
					records may use Test Record of, etc.).
					41-80 Earthquake date and time (including time zone). */
			    /* Maybe one day we'll read Hypoinverse output to detect events, but for now, this is blank */
			fprintf(COSMOS0fp, "Record of                 Earthquake of                                         \n");
			break;
		case 8: /*  Record information:
					8 17-50 Record start time (date and time of first sample; may be approximate - precise value
					should be obtained from real header) and time quality (0 through 5 correspond to
					lowest through highest quality).
					59-80 Record identification, assigned by preparing agency.
				    "06/17/2018, 18:34:38.004 GMT (Q=5) "<- 35 chars   " (Q=5) " "38199368.SB.WLA.00.HN " <- 22char*/ 
			strncpy(record_id, str + 45, 34);
			fprintf(COSMOS0fp, "Rcrd start time:%s %s\n", cos_date, record_id);
			break;
		case 10: /* 10 20-27 Length of raw record, as recorded (seconds); 45-54 Maximum of raw (uncorrected)
					record in g (or other units);* 60-67 Time max occurs in raw record (secs after start). 
					Example:
					Raw record length = 175.340  sec, Uncor max = 100000000, at   40.220 sec. */
			sprintf(tempstr, "Raw record length = %f", (ptrReq->reqEndtime - ptrReq->reqStarttime));
			strncpy(tempstr + 28, "sec, Uncor max =                                    ", 53);
			fprintf(COSMOS0fp, "%s\n", tempstr);
			break;
		default: 
			fprintf(COSMOS0fp, "%s", str);		
		}

	}
	fclose(COSMOS0Libfp);

	/* WRITE Dynamic header; this probably can't be here though because we need to calculate these results */
	/* First we'll write a line that looks like this: */
	/* 17770    raw accel.pts, approx  178 secs, units = counts(50), Format = (7I11) */
	/*Line Cols
	1st 1 - 8 Number of data points following; 10 - 21 Physical parameter of the data.
	35 - 38 Approximate length of record(rounded to nearest sec; see Rhdr(66) for precise value).
	52 - 58 Units of the data values(e.g., cm / sec2); 60 - 61 Index code for units
	*/
	/*sprintf(hour_line, "17770    raw accel.pts, approx  178 secs, units=counts (50),Format=(10I8)\r\n"); /* we want leading spaces */
	/* sprintf(tempstr, "%f", (ptrReq->reqEndtime - ptrReq->reqStarttime) * samplerate_but_we_dont_know_sr_here); */


	if (fwrite("17770    raw accel.pts,   approx 178 secs,  units=counts (50),  Format=(10I8)    \n", 82, 1, COSMOS0fp) != 1)
	{
		logit("et", "COSMOS0PA_next: error writing COSMOS0 dynamic header line. \n");
		return EW_FAILURE;
	}

	return (EW_SUCCESS);
}

/************************************************************************
* This is the working entry point into the disposal system. This        *
* routine gets called for each trace snippet which has been recovered.  *
* It gets to see the corresponding SNIPPET structure, and the event id  *
*                                                                       *
* For PCCOSMOS0, this routine writes to the COSMOS0 file, pointed to by *
* the COSMOS0fp pointer, all of the received trace data in COSMOS0      *
* format:                                                               *
*                                                                       *
*      1. COSMOS0 tag - indicating what follows                         *
*      2. COSMOS0_STATIONCOMP struct - describe the station             *
*      3. COSMOS0 tag - indicating what follows                         *
*      4. COSMOS0_DESCRIPTRACE struct - describe the trace data         *
*      5. trace data                                                    *
*                                                                       *
*  One bit of complexity is that we need to write the files in the      *
*  correct byte-order. Based on the OutputFormat parameter, determine   *
*  whether or not to swap bytes before writing the COSMOS0 file.        *
*                                                                       *
* WARNING: we clip trace data to -2147483648 - +2147483647 so it will   *
*  fit in a long int. Any incoming data that is longer than 32 bits     *
*  will be CLIPPED. cjb 5/18/2001                                       *
*************************************************************************/
/* Process one channel of data */
int COSMOS0PA_next(TRACE_REQ *getThis, double GapThresh,
	long OutBufferLen, int debug,
	int Cadence, double Multiplier)
{
	TRACE2_HEADER *wf;
	char    datatype;
	char    day_line[122];
	char    fourdigits[5];
	char    hour_line[82] = ";                                                                                 "; 
	char    sample[12];
	char    elevendigits[12];
	char   *msg_p;        /* pointer into tracebuf data */
	double  begintime = 0, starttime = 0, endtime = 0, currenttime = 0;
	double  samprate, unrounded;
	float  *f_data;
	int     gap_count = 0;
	int     i, j;
	int     s_place = 0;
	int     tabular_base;
	int     total, raw_counts;
	long    nfill_max = 0l;
	long    nsamp, nfill;
	long    nsamp_this_scn = 0l;
	long    this_size;
	long   *l_data;
	short  *s_data;
	int    current_int;

	/* Put this in the .d file once we know we want it. */
	/*  double multiplier = 0.001; */

	/* Check arguments */
	if (getThis == NULL)
	{
		logit("e", "COSMOS0PA_next: invalid argument passed in.\n");
		return EW_FAILURE;
	}
	/* Used for computing trace statistics */
	total = 0;

	if ((msg_p = getThis->pBuf) == NULL)   /* pointer to first message */
	{
		logit("e", "COSMOS0PA_next: Message buffer is NULL.\n");
		return EW_FAILURE;
	}
	wf = (TRACE2_HEADER *)msg_p;

	/* Look at the first TRACE2_HEADER and get set up of action */
	if (WaveMsg2MakeLocal(wf) < 0)
	{
		logit("e", "COSMOS0PA_next: unknown trace data type: %s\n",
			wf->datatype);
		return(EW_FAILURE);
	}

	nsamp = wf->nsamp;
	starttime = wf->starttime;
	endtime = wf->endtime;
	samprate = wf->samprate;
	if (samprate < 0.0001)
	{
		logit("et", "unreasonable samplerate (%f) for <%s.%s.%s.%s>\n",
			samprate, wf->sta, wf->chan, wf->net, wf->loc);
		return(EW_FAILURE);
	}
	/* LAST header line now that we know sample rate*/
	/* WRITE Dynamic header; this probably can't be here though because we need to calculate these results */
	/* First we'll write a line that looks like this: */
	/* 17770    raw accel.pts, approx  178 secs, units = counts(50), Format = (7I11) */
	/*Line Cols
	1st 1 - 8 Number of data points following; 10 - 21 Physical parameter of the data.
	35 - 38 Approximate length of record(rounded to nearest sec; see Rhdr(66) for precise value).
	52 - 58 Units of the data values(e.g., cm / sec2); 60 - 61 Index code for units
	*/
	/*sprintf(hour_line, "17770    raw accel.pts, approx  178 secs, units=counts (50),Format=(10I8)\r\n"); /* we want leading spaces */
	/* sprintf(tempstr, "%f", (ptrReq->reqEndtime - ptrReq->reqStarttime) * samplerate_but_we_dont_know_sr_here); */

	//raw_counts = int((getThis->reqEndtime - getThis->reqStarttime ) * samprate);
	if (fwrite("17770    raw accel.pts,   approx 178 secs,  units=counts (50),  Format=(10I8)    \n", 82, 1, COSMOS0fp) != 1)





	begintime = starttime;
	/* datatype i4 = intel byte order 4 bytes, s2 = sparc byte order; 2 bytes */
	datatype = 'n';
	if (wf->datatype[0] == 's' || wf->datatype[0] == 'i')
	{
		if (wf->datatype[1] == '2') datatype = 's';
		else if (wf->datatype[1] == '4') datatype = 'l';
	}
	else if (wf->datatype[0] == 't' || wf->datatype[0] == 'f')
	{
		if (wf->datatype[1] == '4') datatype = 'f';
	}
	if (datatype == 'n')
	{
		logit( "et", "COSMOS0PA_next: unsupported datatype: %s\n", wf->datatype );
		return(EW_FAILURE);
	}

	if (debug == 1)
		logit("et", "COSMOS0PA_next: working on <%s/%s/%s/%s> datatype: %c \n",
			wf->sta, wf->chan, wf->net, wf->loc, datatype);

	/********************** loop through all the messages for this s-c-n **********************/
	while (1)
	{
		/* advance message pointer to the data */
		msg_p += sizeof(TRACE2_HEADER);

		/* check for sufficient memory in output buffer */
		this_size = (nsamp_this_scn + nsamp) * sizeof(long);
		if (OutBufferLen < this_size)
		{
			logit("e", "out of space for <%s.%s.%s.%s>; saving long trace.\n",
				wf->sta, wf->chan, wf->net, wf->loc);
			break;
		}

		/* if data are floats, clip to longs cjb 5/18/2001 */
		switch (datatype)
		{
		case 's':
			s_data = (short *)msg_p;
			for (j = 0; j < nsamp; j++, nsamp_this_scn++)
				COSMOS0Buffer[nsamp_this_scn] = (long)s_data[j];
			msg_p += sizeof(short) * nsamp;
			break;
		case 'l':
			l_data = (long *)msg_p;
			for (j = 0; j < nsamp; j++, nsamp_this_scn++)
				COSMOS0Buffer[nsamp_this_scn] = l_data[j];
			msg_p += sizeof(long) * nsamp;
			break;
		case 'f':
			f_data = (float *)msg_p;
			/* CLIP the data to long int */
			for (j = 0; j < nsamp; j++, nsamp_this_scn++)
			{
				if (l_data[j] < (float)LONG_MIN)
					COSMOS0Buffer[nsamp_this_scn] = LONG_MIN;
				else if (l_data[j] > (float)LONG_MAX)
					COSMOS0Buffer[nsamp_this_scn] = LONG_MAX;
				else
					COSMOS0Buffer[nsamp_this_scn] = (long)l_data[j];
			}
			msg_p += sizeof(float) * nsamp;
			break;
		}

		/* msg_p has been advanced to the next TRACE_BUF; localize bytes *
		* and check for gaps.                                            */
		wf = (TRACE2_HEADER *)msg_p;
		if (WaveMsg2MakeLocal(wf) < 0)
		{
			if (debug == 1)
				logit("e", "COSMOS0PA_next: unknown trace data type or unexpected end of data: %s\n",
					wf->datatype);
			else
				logit("e", "COSMOS0PA_next: unknown trace data type or unexpected end of data.\n");
			break;
			//return(EW_FAILURE);
		}
		nsamp = wf->nsamp;
		starttime = wf->starttime;
		/* starttime is set for new packet; endtime is still set for old packet */
		if (endtime + (1.0 / samprate) * GapThresh < starttime)
		{
			/* there's a gap, so fill it */
			if (debug == 1)
				logit("e", "gap in %s.%s.%s.%s: %lf: %lf\n", wf->sta, wf->chan, wf->net,
					wf->loc, endtime, starttime - endtime);
			nfill = (long)(samprate * (starttime - endtime) - 1);
			if ((nsamp_this_scn + nfill) * (long)sizeof(long) > OutBufferLen)
			{
				logit( "e", "bogus gap (%ld); skipping\n", nfill );
				return(EW_FAILURE);
			}
			/* do the filling */
			for (j = 0; j < nfill; j++, nsamp_this_scn++)
				COSMOS0Buffer[nsamp_this_scn] = getThis->fill; // changed from local variable fill swl
														   /* keep track of how many gaps and the largest one */
			gap_count++;
			if (nfill_max < nfill)
				nfill_max = nfill;
		}
		/* Advance endtime to the new packet;        *
		* process this packet in the next iteration */
		endtime = wf->endtime;
	} /******************** end while(1) ***************************************************/

	  /* If the sample rate is 1 sample per minute then we'll have a sample rate of .016 */
	  /* For minute data we want 24 rows of 60 samples each, 1440 samples in a day. */
	  /* A single file fills the month of October for SJG, and includes four trace types */
	  /* FYXZ, so there are a total of 2976 lines in this file. */

	  /* Match our metadata with our waveserver request, if possible.
	  * observatory's value is -1 if there isn't a match. */

	currenttime = begintime;
	j = 0;
	if (Cadence == MINUTE) {

		while ((j < nsamp_this_scn) && (currenttime < getThis->reqEndtime))
		{
			/* Only give them what they asked for, not each sample we got back.
			Tracebufs contain multiple samples, and we may need to request
			an earlier one to get the start sample we need, or a later one
			for the last sample*/
			while (currenttime < getThis->reqStarttime) {
				currenttime = currenttime + 1 / wf->samprate;
				j++;
			}

			s_place = 0;

			/* 35-394 60I6   60 6-digit 1-minute values for the given element for that data hour.
			*               The values are in tenth-minutes for D and I, and in
			*               nanoTeslas for the intensity elements.
			*/
			total = 0;
			LineMean = 0;

			i = 0;

			/* WRITE DATA */

			/* Drop 60 sample values into the line while counting up the total for  */
			/* later calculating the mean of these 60 samples. If there is any gap  */
			/* we won't bother with the mean, LineMean becomes 9999                 */
			while (i < 7 && j < nsamp_this_scn)
			{
				if (COSMOS0Buffer[j] != getThis->fill) {
					sprintf(elevendigits, "           "); /* we want leading spaces */
					if (((int)(COSMOS0Buffer[j] * Multiplier) > 999999) || ((int)(COSMOS0Buffer[j] * Multiplier) < -99999)) {
						sprintf(sample, "9999");
						/* prevent out of range string */
					}
					else {
						sprintf(sample, "%d", (int)(COSMOS0Buffer[j] * Multiplier));
					}
					strcpy(elevendigits + 11 - strlen(sample), sample);
					strcpy(hour_line + s_place, elevendigits);
				}
				else {
					/* We have a gap */
					strcpy(hour_line + s_place, "  9999");
					LineMean = 9999;
				}
				s_place = s_place + 11;

				total += (int)(COSMOS0Buffer[j] * Multiplier);

				j++; i++;
			}


			/* 401-402       Record end marker.
			*               Two chars 'cr'= 13 and 'nl'= 10.
			*/
			hour_line[80] = '\r';
			hour_line[80] = ' ';
			hour_line[81] = '\n';

			/* Write out line */
			if (fwrite(&hour_line, sizeof(hour_line), 1, COSMOS0fp) != 1)
			{
				logit("et", "COSMOS0PA_next: error writing COSMOS0 line. \n");
				return EW_FAILURE;
			}
		}
	}/* End if (Cadence == MINUTE) */
	 /************************* HOUR **************************************************/
	else if (Cadence == HOUR) {

		/* Skip ahead to the time range requested */
		while ((currenttime < getThis->reqStarttime) && (j < nsamp_this_scn)) {
			currenttime = currenttime + 1 / wf->samprate;
			j++;
		}

		/* We first try and find a tabular base that will apply to all data
		* that has been returned from our waveserver request. If there is
		* no base that will fit all the data and be within -999 to 9998, we will
		* keep removing low and high values 'till we get one. This base will work
		* for most days, but not spike days. Spike days will recalculate their
		* own tabular base; We'll skip extra data at the beginning FIX: but not at end */
		tabular_base = FindTabularBase(nsamp_this_scn, COSMOS0Buffer, getThis, Multiplier, j);

		while ((j < nsamp_this_scn) && (currenttime < getThis->reqEndtime)) {

			/* On to data */

			/*    21-116    24I4     24 4-digit hourly mean values for the day.		*/
			/*                       The values are in tenth-minutes for D and I, and in	*/
			/*                       nanoTeslas for the intensity elements.		 	*/
			/*                       The first hourly mean value represents the mean value	*/
			/*                       between 00:00 UT and 01:00 UT, ..., the 24th value	*/
			/*                       represents the mean between 23:00 UT and 24:00 UT.	*/
			/*                       A missing value is identified by 9999.		 	*/
			s_place = 0;
			total = 0;
			LineMean = 0;
			i = 0;

			/* WRITE DATA */
			/* Drop 24 sample values into the line while counting up the total for
			* later calculating the mean of these 24 samples. If there is any gap
			* we won't bother with the mean, LineMean becomes 9999                 */
			while (i < 24 && j < nsamp_this_scn)
			{
				unrounded = ((COSMOS0Buffer[j] * Multiplier)) - (tabular_base * 100);
				if (unrounded < 0) {
					current_int = (int)(unrounded - .5);
				}
				else {
					current_int = (int)(unrounded + .5);
				}
				if ((current_int > 9999) || (current_int < -999)) {
					/* How did this happen? */
					if (debug == 1) {
						logit("et", "current_int out of range for COSMOS0 hourly format %d\n", current_int);
					}
					current_int = 9999;
				}
				/*current_int = (int)((COSMOS0Buffer[j]*Multiplier)) - (tabular_base*100);*/
				if (COSMOS0Buffer[j] != getThis->fill) {
					sprintf(fourdigits, "    "); /* we want leading spaces */;
					sprintf(sample, "%4d", current_int);
					strcpy(fourdigits + 4 - strlen(sample), sample);
					strcpy(day_line + s_place, fourdigits);
					total += current_int;

				}
				else {
					/* We have a gap */
					strcpy(day_line + s_place, "9999");
					LineMean = 9999;
				}
				s_place = s_place + 4;

				currenttime = currenttime + 1 / wf->samprate;
				if (currenttime >= getThis->reqEndtime) {
					//break;
					j = nsamp_this_scn;
				}
				j++; i++;
			}
			while (i < 24) {
				/* If we got here, and i is less than 24, then we've got no more samples,
				* but we're not finished with an hour line yet.
				* Fill this line with COSMOS0 gap fill values. */
				strcpy(day_line + s_place, "9999");
				LineMean = 9999;
				s_place = s_place + 4;
				i++;
			}

			/*    117-120   I4       Daily Mean.		 */
			/*                       If any of the hourly mean values for the day are missing		 */
			/*                       9999 will appear as the daily mean.		 */
			if ((LineMean > 9999) || (LineMean < -999)) {
				LineMean = 9999;
				/* prevent out of range string */
			}
			if (LineMean != 9999) {
				LineMean = total / 24;  /* Mean of the first 24 samples if there are no gaps */
										/* We've already subtracted the tabular base from each */
										/* value before dropping in the total. */
			}
			sprintf(fourdigits, "    "); /* we want leading spaces */
			sprintf( sample, "%ld", LineMean );
			strcpy(fourdigits + 4 - strlen(sample), sample);
			strcpy(day_line + 116, fourdigits);

			/*    121-122            Record end marker.		 */
			/*                       Two chars 'cr'= 13 and 'nl'= 10.		 */
			day_line[120] = '\r';
			day_line[121] = '\n';
			/* Write out line */
			if (fwrite(&day_line, sizeof(day_line), 1, COSMOS0fp) != 1)
			{
				logit("et", "COSMOS0PA_next: error writing COSMOS0 line. \n");
				return EW_FAILURE;
			}

		} /* End whiling through the data samples */
	} /* End if (Cadence == HOUR) */
	return EW_SUCCESS;
}

/************************************************************************
* This is the Put Away end event routine. It's called after we've       *
* finished processing one event                                         *
*                                                                       *
* For PC-COSMOS0 - close the COSMOS0 file, pointed to by COSMOS0fp      *
*               free COSMOS0Buffer memory to be nice to everyone else   *
*************************************************************************/
int COSMOS0PA_end_ev(int debug)
{
	fclose(COSMOS0fp);

	if (debug == 1)
		logit("t", "Closing COSMOS0 file \n");

	return(EW_SUCCESS);
}

/************************************************************************
*       This is the Put Away close routine. It's called after when      *
*       we're being shut down.                                          *
*************************************************************************/
int COSMOS0PA_close(int debug)
{

	free((char *)COSMOS0Buffer);
	return(EW_SUCCESS);
}

/************************************************************************
* We first try and find a tabular base that will apply to all data      *
* that has been returned from our waveserver request. If there is       *
* no base that will fit all the data and be within -999 to 9998, we     *
* will keep removing low and high values 'till we get one. This base    *
* will work for most days, but not spike days. Spike days will          *
* recalculate their own tabular base                                    *
*************************************************************************/
int FindTabularBase(int number_samples, long *SampleBuffer, TRACE_REQ *getThis, double Multiplier, int skip)
{
	int daymax;
	int daymin;
	int tabular_base = 999;
	int range;
	int j;
	int sample_COSMOS0_int;

	daymax = (int)getThis->fill;
	daymin = (int)getThis->fill;

	for (j = skip; j<number_samples; j++) {
		/* From the definition above: For the intensity elements we have that
		* hourly value (nanoTeslas) = tab.base*100 + tab.value
		* The Multiplier is specific to the USGS waveservers so they can get
		* decimal values out of the integer type...                             */
		/* Won't divide by 100 'till we're done */

		/* Skip ahead past all gaps */
		if (SampleBuffer[j] == getThis->fill) {
			continue;
		}
		sample_COSMOS0_int = (int)((SampleBuffer[j] * Multiplier));
		/* Keep track of the line's max and min for the tabular base generation */
		if ((sample_COSMOS0_int > daymax) || (daymax == getThis->fill)) {
			daymax = sample_COSMOS0_int;
		}
		if ((sample_COSMOS0_int < daymin) || (daymin == getThis->fill)) {
			daymin = sample_COSMOS0_int;
		}
		range = daymax - daymin;
	}
	/* Really we want our values ideally to be less than 1000 and > than 1.
	* second most ideally less than 1000 and > than -999.
	* after that, not ideal, anything that fits between 9998 and -999
	* after that, grab the median and make that 5500;
	* everything out of range becomes a spike = 9999 */
	if (range < 899) {
		/* let's have the lowest number in our range = 0 *
		* so we will subtract that number from all numbers. */
		tabular_base = daymin / 100;
	}
	else if (range < 1899) {
		/* let's have the highest number in our range = 999
		* so we have as few negative numbers as possible */
		tabular_base = (daymax - 900) / 100;
	}
	else if (range < (9898 + 999 + 1)) { /* the 1 is to account for 0; 999 is for to -999*/
										 /* let's have the lowest number in our range = -999
										 * so our numbers are as low as possible, and are more
										 * likely to be less than 3 digits . */
		tabular_base = (daymin + 999) / 100;
	}
	else if (number_samples < 25) {
		/* out of range: if we're out of range then let's make the
		* median value equal to 5500. We'll have to drop (ie: make 9999)
		* spike numbers on this line that are > 9998 and < -999  */
		tabular_base = (Median(number_samples, SampleBuffer) + 5500) / 100;
	}
	else {
		/* If we're finding the tabular base for a larger range than a day
		* and we can't find one, don't use the median, just return failure
		* We might be doing this we we want to find a common base for an
		* entire COSMOS0 file for a particular element. So if this fails,
		* First line will determine it's own tabular base. Each subsequent
		* line will try to use the previous line's base. If that fails the
		* subsequent line will calculate a new base. */
		tabular_base = 9999;
	}
	return(tabular_base);
}

/*
*
*  Byte swapping functions
*/

/* swap bytes if necessary, depending on what machine */
/* we have been compiled, and what machine the data   */
/* is being read from/being written to                */

static int StructMakeLocal(void *ptr, int struct_type, char data_type,
	int debug)
{
	if (ptr == NULL)
	{
		logit("e", "StructMakeLocal: NULL pointer passed in\n");
		return EW_FAILURE;
	}

#if defined (_INTEL)
	if (data_type != '6')
	{
		if (debug == 1)
			logit("et", "Hoping we don't really have to do any swapping from Intel to Sun \n");

	}

#elif defined (_SPARC)
	if (data_type == '6')
	{
		if (debug == 1)
			logit("et", "Hoping we don't really have to do any swapping from Sun to Intel because we've deleted from COSMOS0 the SwapDo function that suds used. \n");
	}

#endif
	return EW_SUCCESS;
}

long Median(int number_of_array_elements, long *Array)
{
	long *LocalArray;

	LocalArray = (long*)malloc(number_of_array_elements * sizeof(long));

	qsort(&LocalArray[0], number_of_array_elements, sizeof(long), longCompareForQsort);
	/* Get Median */
	return(LocalArray[(long)(number_of_array_elements / 2)]);
}
/*****************************************************************/
/* Just a simple compare function so that Q sort does it's thing */
/*****************************************************************/
int longCompareForQsort(const void *x, const void *y)
{
	/* compare must have const void ptr params for qsort to work */
	const long   *ix = (const long *)x;
	const long   *iy = (const long *)y;
	/* returns 1, -1 or 0 */
	return (*ix > *iy) - (*ix < *iy);
}
