package pickewanalysis.wavesources;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.InputStreamReader;
import java.net.Socket;
import java.util.ArrayList;
import java.util.Locale;
import pickewanalysis.waveform.WaveFormGroup;

public class WaveServer extends WaveSource {

    private static int requestID = 1;
    private Socket socket;
    private DataInputStream Sinput;
    private DataOutputStream Soutput;
    public int TimeOutInterval = 5000; //Waveserver timeout
    public int ReadMargin = 30000; //Margin to read before the requested start

    public WaveServer() {
        //Do nothing
    }

    @Override
    public void update() throws Exception {
        this.clear();
        WSMenu wsm = getMenu();
        if (wsm == null) {
            return;
        }
        for (WSMenuItem item : wsm) {
            WaveSourceStation station = new WaveSourceStation(
                    item.getStation(), item.getChannel(),
                    item.getNetwork(), item.getLocation(),
                    item.getStartTime(),
                    item.getEndTime());
            this.add(station);
        }
    }

    public String getIP() {
        String[] parts = this.sourceName.split("[:]+");
        return parts[0];
    }

    public int getPort() {
        String[] parts = this.sourceName.split("[:]+");
        return Integer.valueOf(parts[1]);
    }

    @Override
    public WaveFormGroup read(String stat, String chan, String net, String loc,
            long starttime, long endtime) throws Exception {
        this.getMenu();

        //Start communications
        connect();

        //Send request
        //GETSCNLRAW: <request id> <s><c><n><l> <starttime> <endtime>
        int curReqID = getReqID();
        String request = String.format(Locale.UK, "GETSCNLRAW: %d %s %s %s %s %8.6f %8.6f SCNL\n",
                curReqID, stat, chan, net, loc, (double) (starttime-ReadMargin) / 1000, (double) (endtime+ReadMargin) / 1000);
        Soutput.write(request.getBytes());
        Soutput.flush();

        //Check for response header
        String response = "";
        byte b;
        while ((b = Sinput.readByte()) != (byte) 10) {
            response += (char) b;
        }
        String[] parts = null;
        parts = response.trim().split("[\\s]+");
        //Check for data availability
        if (parts == null || !parts[6].equals("F")) {
            close();
            return null;
        }
        //Read raw data
        int nbytes = Integer.valueOf(parts[10]);
        int counter = 0;
        int ntries = 0;
        byte[] rawdata = new byte[nbytes];
        try {
            while (counter < nbytes && ntries < 10) {
                counter += Sinput.read(rawdata, counter, nbytes - counter);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        close();
        if (counter != nbytes) {
            return null;
        }

        //Process raw data
        WaveFormGroup wfg = WaveFormGroup.parseTraceBuf2(rawdata);
        //System.out.println(wfg);
        return wfg;
    }

    private WSMenu getMenu() throws Exception {
        //Start communications
        connect();
        //Send menu request
        int curReqID = getReqID();
        String request = String.format(Locale.UK, "MENU: %d SCNL\n", curReqID);
        Soutput.write(request.getBytes());
        Soutput.flush();
        //Check for response
        BufferedReader Binput = new BufferedReader(new InputStreamReader(Sinput));
        String[] parts = null;
        try {
            do {
                String response = Binput.readLine();
                //System.out.println(response);
                parts = response.trim().split("[\\s]+");
            } while (Integer.valueOf(parts[0]) != curReqID);
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (parts == null) {
            close();
            return null;
        }
        int nsta = (parts.length - 1) / 8;
        if (nsta == 0) {
            //Close comms
            close();
            return null;
        }

        //Store menu
        WSMenu menu = new WSMenu(nsta);
        for (int i = 0; i < nsta; i++) {
            int p = i * 8 + 2; //Startting position, skipping the pin number
            WSMenuItem wsmi = new WSMenuItem(parts[p + 0], parts[p + 1], parts[p + 2], parts[p + 3],
                    (long) (Double.valueOf(parts[p + 4]) * 1000),
                    (long) (Double.valueOf(parts[p + 5]) * 1000 + 1));
            menu.add(wsmi);
        }

        //Close comms
        close();

        return menu;
    }

    private void connect() throws Exception {
        //Start socket
        socket = new Socket(this.getIP(), this.getPort());
        socket.setSoTimeout(TimeOutInterval);
        //Start streams
        Sinput = new DataInputStream(socket.getInputStream());
        Soutput = new DataOutputStream(socket.getOutputStream());
    }

    private void close() throws Exception {
        Sinput.close();
        Soutput.close();
        socket.close();
    }

    private int getReqID() {
        return requestID++;
    }
}

class WSMenu extends ArrayList<WSMenuItem> {

    WSMenu(int capacity) {
        super(capacity);
    }
}

class WSMenuItem {

    String stat, chan, net, loc;
    long starttime;
    long endtime;

    WSMenuItem(String stat, String chan, String net, String loc, long starttime, long endtime) {
        this.stat = stat;
        this.chan = chan;
        this.net = net;
        this.loc = loc;
        this.starttime = starttime;
        this.endtime = endtime;
    }

    String getStation() {
        return stat;
    }

    String getChannel() {
        return chan;
    }

    String getNetwork() {
        return net;
    }

    String getLocation() {
        return loc;
    }

    long getStartTime() {
        return starttime;
    }

    long getEndTime() {
        return endtime;
    }

    @Override
    public String toString() {
        return String.format(Locale.UK, "%s.%s.%s.%s %16.6f %16.6f",
                stat, chan, net, loc,
                (double) starttime / 1000, (double) endtime / 1000);
    }
}
