/*
 * Decompiled with CFR 0.152.
 */
package gov.usgs.plot.data;

import gov.usgs.math.Butterworth;
import gov.usgs.math.FFT;
import gov.usgs.math.Filter;
import gov.usgs.plot.data.BinaryDataSet;
import gov.usgs.util.Log;
import gov.usgs.util.Util;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.TreeMap;
import java.util.logging.Logger;

public class Wave
implements BinaryDataSet,
Comparable<Wave>,
Cloneable {
    protected static final Logger logger = Log.getLogger("gov.usgs.vdx.data.wave.Wave");
    public static int NO_DATA = Integer.MIN_VALUE;
    public int[] buffer;
    private double startTime;
    private double samplingRate;
    private double registrationOffset = Double.NaN;
    protected static final long TO_USEC = 1000000L;
    protected static final double FROM_USEC = 1.0E-6;
    private transient double mean = Double.NaN;
    private transient double rsam = Double.NaN;
    private transient int max = Integer.MIN_VALUE;
    private transient int min = Integer.MAX_VALUE;
    private transient int[] dataRange = null;
    private transient int first = NO_DATA;
    private String dataType;

    public Wave() {
    }

    public Wave(Wave wave) {
        this.startTime = wave.startTime;
        this.samplingRate = wave.samplingRate;
        this.buffer = new int[wave.buffer.length];
        System.arraycopy(wave.buffer, 0, this.buffer, 0, this.buffer.length);
    }

    public Wave(int[] b, double st, double sr) {
        this.makeWave(b, st, sr, "s4");
    }

    public Wave(int[] b, double st, double sr, String dt) {
        this.makeWave(b, st, sr, dt);
    }

    public void makeWave(int[] b, double st, double sr, String dt) {
        this.buffer = b;
        this.startTime = st;
        this.samplingRate = sr;
        this.dataType = dt;
    }

    public Wave(ByteBuffer bb) {
        this.fromBinary(bb);
    }

    public void register() {
        double dif = this.startTime % this.getSamplingPeriod();
        this.registrationOffset = dif >= this.getSamplingPeriod() / 2.0 ? this.getSamplingPeriod() - dif : -dif;
        this.startTime += this.registrationOffset;
    }

    public void setRegistrationOffset(double o) {
        this.registrationOffset = o;
    }

    public boolean isData() {
        for (int i = 0; i < this.buffer.length; ++i) {
            if (this.buffer[i] == NO_DATA) continue;
            return true;
        }
        return false;
    }

    public void handleBadData() {
        for (int i = 0; i < this.buffer.length; ++i) {
            if (this.buffer[i] != 999999) continue;
            this.buffer[i] = NO_DATA;
        }
    }

    public void trunc(int s) {
        int[] buf = new int[s];
        System.arraycopy(this.buffer, 0, buf, 0, s);
        this.buffer = buf;
    }

    public void subtract(int m) {
        int i = 0;
        while (i < this.buffer.length) {
            int n = i++;
            this.buffer[n] = this.buffer[n] - m;
        }
    }

    public void setSamplingRate(double sr) {
        this.samplingRate = sr;
    }

    public void setDataType(String dt) {
        this.dataType = dt;
    }

    public void setStartTime(double st) {
        this.startTime = st;
    }

    public double[][] fft() {
        int i;
        int n = this.buffer.length;
        int p2 = (int)Math.ceil(Math.log(n) / Math.log(2.0));
        int newSize = (int)Math.pow(2.0, p2);
        double[][] buf = new double[newSize][2];
        int m = (int)Math.round(this.mean());
        for (i = 0; i < n; ++i) {
            if (this.buffer[i] == NO_DATA) continue;
            buf[i][0] = this.buffer[i];
        }
        for (i = n; i < newSize; ++i) {
            buf[i][0] = m;
        }
        FFT.fft(buf);
        return buf;
    }

    public void invalidateStatistics() {
        this.mean = Double.NaN;
        this.rsam = Double.NaN;
        this.max = Integer.MIN_VALUE;
        this.min = Integer.MAX_VALUE;
        this.first = NO_DATA;
    }

    private void deriveStatistics() {
        if (this.buffer == null || this.buffer.length == 0) {
            this.mean = 0.0;
            this.rsam = 0.0;
            this.max = 0;
            this.min = 0;
            this.first = 0;
            return;
        }
        int noDatas = 0;
        long sum = 0L;
        long rs = 0L;
        boolean firstSet = false;
        for (int i = 0; i < this.buffer.length; ++i) {
            int d = this.buffer[i];
            if (d != NO_DATA) {
                sum += (long)d;
                rs += (long)Math.abs(d);
                this.min = Math.min(this.min, d);
                this.max = Math.max(this.max, d);
                if (firstSet) continue;
                this.first = d;
                firstSet = true;
                continue;
            }
            ++noDatas;
        }
        this.mean = (double)sum / (double)(this.buffer.length - noDatas);
        this.rsam = (double)rs / (double)(this.buffer.length - noDatas);
        this.dataRange = new int[]{this.min, this.max};
    }

    public int first() {
        if (this.first == NO_DATA) {
            this.deriveStatistics();
        }
        return this.first;
    }

    public double mean() {
        if (Double.isNaN(this.mean)) {
            this.deriveStatistics();
        }
        return this.mean;
    }

    public int max() {
        if (this.max == Integer.MIN_VALUE) {
            this.deriveStatistics();
        }
        return this.max;
    }

    public int min() {
        if (this.min == Integer.MAX_VALUE) {
            this.deriveStatistics();
        }
        return this.min;
    }

    public double rsam() {
        if (Double.isNaN(this.rsam)) {
            this.deriveStatistics();
        }
        return this.rsam;
    }

    public void decimate(int factor) {
        int[] buf = new int[this.numSamples() / factor];
        for (int i = 0; i < this.numSamples() / factor; ++i) {
            buf[i] = this.buffer[i * factor];
        }
        this.buffer = buf;
        this.samplingRate /= (double)factor;
    }

    public int numSamples() {
        return this.buffer.length;
    }

    public int numNoData() {
        int count = 0;
        for (int sample : this.buffer) {
            if (sample != NO_DATA) continue;
            ++count;
        }
        return count;
    }

    public void convertToJ2K() {
        this.startTime = Util.ewToJ2K(this.startTime);
    }

    public String getDataType() {
        return this.dataType;
    }

    public double getSamplingRate() {
        return this.samplingRate;
    }

    public double getSamplingPeriod() {
        return 1.0 / this.samplingRate;
    }

    public double getNyquist() {
        return this.samplingRate / 2.0;
    }

    public double getStartTime() {
        return this.startTime;
    }

    public double getEndTime() {
        return this.startTime + (double)this.numSamples() * this.getSamplingPeriod();
    }

    public double getRegistrationOffset() {
        return this.registrationOffset;
    }

    public int[] getDataRange() {
        if (this.min == Integer.MAX_VALUE) {
            this.deriveStatistics();
        }
        return this.dataRange;
    }

    public boolean adjacent(Wave wave) {
        double end;
        double start = Math.abs(this.getStartTime() - wave.getEndTime());
        return Math.min(start, end = Math.abs(this.getEndTime() - wave.getStartTime())) <= this.getSamplingPeriod() * 1.25;
    }

    public boolean overlaps(Wave wave) {
        if (this.getEndTime() <= wave.getStartTime()) {
            return false;
        }
        return !(this.getStartTime() >= wave.getEndTime());
    }

    public boolean overlaps(double t1, double t2) {
        if (this.getEndTime() <= t1) {
            return false;
        }
        return !(this.getStartTime() >= t2);
    }

    public Wave[] split() {
        Wave sw1 = new Wave();
        Wave sw2 = new Wave();
        sw1.startTime = this.startTime;
        sw1.samplingRate = this.samplingRate;
        int length1 = this.buffer.length / 2;
        sw1.buffer = new int[length1];
        System.arraycopy(this.buffer, 0, sw1.buffer, 0, length1);
        sw1.dataType = this.dataType;
        sw2.startTime = this.startTime + (double)length1 * this.getSamplingPeriod();
        sw2.samplingRate = this.samplingRate;
        int length2 = this.buffer.length / 2 + this.buffer.length % 2;
        sw2.buffer = new int[length2];
        System.arraycopy(this.buffer, length1, sw2.buffer, 0, length2);
        sw2.dataType = this.dataType;
        return new Wave[]{sw1, sw2};
    }

    public List<Wave> split(int maxSamples) {
        int numSamples;
        ArrayList<Wave> list = new ArrayList<Wave>(this.buffer.length / maxSamples + 2);
        double ct = this.startTime;
        for (int j = 0; j < this.buffer.length; j += numSamples) {
            numSamples = Math.min(maxSamples, this.buffer.length - j);
            Wave sw = new Wave();
            sw.startTime = ct;
            sw.samplingRate = this.samplingRate;
            sw.dataType = this.dataType;
            sw.buffer = new int[numSamples];
            System.arraycopy(this.buffer, j, sw.buffer, 0, numSamples);
            ct += (double)numSamples * this.getSamplingPeriod();
            list.add(sw);
        }
        return list;
    }

    public Wave subset(double t1, double t2) {
        if (t1 < this.getStartTime() || t2 > this.getEndTime() || t2 < t1) {
            return this;
        }
        int samples = (int)Math.floor((t2 - t1) * this.samplingRate);
        int offset = (int)Math.floor((t1 - this.startTime) * this.samplingRate);
        Wave sw = new Wave();
        sw.startTime = t1;
        sw.samplingRate = this.samplingRate;
        sw.dataType = this.dataType;
        sw.registrationOffset = this.registrationOffset;
        sw.buffer = new int[samples];
        System.arraycopy(this.buffer, offset, sw.buffer, 0, samples);
        return sw;
    }

    public void erase(double t1, double t2) {
        int[] buf;
        int size;
        if (t2 < this.getStartTime() || t1 > this.getEndTime()) {
            return;
        }
        if (t1 >= this.getStartTime() && t2 <= this.getEndTime()) {
            return;
        }
        if (t1 <= this.getStartTime() && t2 >= this.getEndTime()) {
            this.buffer = null;
            this.startTime = Double.NaN;
            this.samplingRate = Double.NaN;
            this.dataType = null;
        }
        if (t2 > this.getStartTime() && t2 <= this.getEndTime()) {
            size = (int)Math.round((t2 - this.getStartTime()) * this.samplingRate);
            buf = new int[this.buffer.length - size];
            System.arraycopy(this.buffer, size, buf, 0, this.buffer.length - size);
            this.buffer = buf;
            this.startTime = t2;
        }
        if (t1 >= this.getStartTime() && t1 <= this.getEndTime()) {
            size = (int)Math.round((this.getEndTime() - t1) * this.samplingRate);
            buf = new int[this.buffer.length - size];
            System.arraycopy(this.buffer, 0, buf, 0, this.buffer.length - size);
            this.buffer = buf;
        }
    }

    public Wave combine(Wave wave) {
        Wave rightWave;
        Wave leftWave;
        if (this.samplingRate != wave.getSamplingRate() || !this.adjacent(wave) && !this.overlaps(wave)) {
            return null;
        }
        if (this.startTime >= wave.getStartTime() && this.getEndTime() <= wave.getEndTime()) {
            return wave;
        }
        if (this.startTime <= wave.getStartTime() && this.getEndTime() >= wave.getEndTime()) {
            return this;
        }
        double len = Math.max(this.getEndTime(), wave.getEndTime()) - Math.min(this.startTime, wave.getStartTime());
        int newLength = (int)Math.ceil(len *= this.samplingRate);
        int totalSamples = wave.numSamples() + this.numSamples();
        newLength = Math.min(newLength, totalSamples);
        int[] newbuf = new int[newLength];
        if (this.startTime <= wave.getStartTime()) {
            leftWave = this;
            rightWave = wave;
        } else {
            leftWave = wave;
            rightWave = this;
        }
        System.arraycopy(leftWave.buffer, 0, newbuf, 0, leftWave.buffer.length);
        int i = newbuf.length - rightWave.buffer.length;
        System.arraycopy(rightWave.buffer, 0, newbuf, i, rightWave.buffer.length);
        this.buffer = newbuf;
        this.startTime = leftWave.startTime;
        return this;
    }

    public static Wave join(List<Wave> waves) {
        if (waves == null || waves.size() == 0) {
            return null;
        }
        if (waves.size() == 1) {
            return waves.get(0);
        }
        double mint = 1.0E300;
        double maxt = -1.0E300;
        double sr = -1.0;
        for (Wave sw : waves) {
            mint = Math.min(mint, sw.getStartTime());
            maxt = Math.max(maxt, sw.getEndTime());
            sr = sw.getSamplingRate();
        }
        int samples = (int)((maxt - mint) * sr);
        int[] buffer = new int[samples + 1];
        Arrays.fill(buffer, NO_DATA);
        for (Wave sw : waves) {
            int i = (int)Math.round((sw.getStartTime() - mint) * sr);
            System.arraycopy(sw.buffer, 0, buffer, i, sw.buffer.length);
        }
        return new Wave(buffer, mint, sr);
    }

    public static Wave join(List<Wave> waves, double t1, double t2) {
        if (waves == null || waves.size() == 0) {
            return null;
        }
        Wave wv0 = waves.get(0);
        int samples = (int)((t2 - t1) * wv0.getSamplingRate());
        int[] buffer = new int[samples + 1];
        Arrays.fill(buffer, NO_DATA);
        for (Wave sw : waves) {
            int i = (int)Math.round((sw.getStartTime() - t1) * wv0.getSamplingRate());
            System.arraycopy(sw.buffer, 0, buffer, i, sw.buffer.length);
        }
        return new Wave(buffer, t1, wv0.getSamplingRate());
    }

    public int getMemorySize() {
        if (this.buffer == null) {
            return 0;
        }
        return this.buffer.length * 4;
    }

    public double[] integrate() {
        double[] d = new double[this.buffer.length];
        double period = this.getSamplingPeriod();
        double mean = this.mean();
        double sum = 0.0;
        for (int i = 0; i < this.buffer.length; ++i) {
            d[i] = (sum += (double)this.buffer[i] - mean) * period;
        }
        return d;
    }

    public double[] removeMean() {
        double[] d = new double[this.buffer.length];
        double mean = this.mean();
        for (int i = 0; i < this.buffer.length; ++i) {
            d[i] = (double)this.buffer[i] - mean;
        }
        return d;
    }

    public void filter(Butterworth bw, boolean zeroPhaseShift) {
        int i;
        double mean = this.mean();
        double[] dBuf = new double[this.buffer.length + (int)((double)this.buffer.length * 0.5)];
        Arrays.fill(dBuf, mean);
        int trueStart = (int)((double)this.buffer.length * 0.25);
        for (i = 0; i < this.buffer.length; ++i) {
            if (this.buffer[i] == NO_DATA) continue;
            dBuf[i + trueStart] = this.buffer[i];
        }
        bw.setSamplingRate(this.getSamplingRate());
        bw.create();
        Filter.filter(dBuf, bw.getSize(), bw.getXCoeffs(), bw.getYCoeffs(), bw.getGain(), 0.0, 0.0);
        if (zeroPhaseShift) {
            double[] dBuf2 = new double[dBuf.length];
            int i2 = 0;
            int j = dBuf.length - 1;
            while (i2 < dBuf.length) {
                dBuf2[j] = dBuf[i2];
                ++i2;
                --j;
            }
            Filter.filter(dBuf2, bw.getSize(), bw.getXCoeffs(), bw.getYCoeffs(), bw.getGain(), 0.0, 0.0);
            i2 = 0;
            j = dBuf2.length - 1 - trueStart;
            while (i2 < this.buffer.length) {
                this.buffer[i2] = (int)Math.round(dBuf2[j]);
                ++i2;
                --j;
            }
        } else {
            for (i = 0; i < this.buffer.length; ++i) {
                this.buffer[i] = (int)Math.round(dBuf[i + trueStart]);
            }
        }
        this.invalidateStatistics();
    }

    @Override
    public ByteBuffer toBinary() {
        ByteBuffer bb = ByteBuffer.allocate(28 + 4 * this.buffer.length + 4);
        bb.putDouble(this.startTime);
        bb.putDouble(this.samplingRate);
        bb.putDouble(this.registrationOffset);
        bb.putInt(this.buffer.length);
        for (int i = 0; i < this.buffer.length; ++i) {
            bb.putInt(this.buffer[i]);
        }
        if (this.dataType != null) {
            bb.putChar(this.dataType.charAt(0));
            bb.putChar(this.dataType.charAt(1));
        }
        return bb;
    }

    @Override
    public void fromBinary(ByteBuffer bb) {
        this.startTime = bb.getDouble();
        this.samplingRate = bb.getDouble();
        this.registrationOffset = bb.getDouble();
        int length = bb.getInt();
        this.buffer = new int[length];
        for (int i = 0; i < length; ++i) {
            this.buffer[i] = bb.getInt();
        }
        try {
            char[] ca = new char[]{bb.getChar(), bb.getChar()};
            this.dataType = new String(ca);
        }
        catch (Exception e) {
            logger.fine("Extracting dataType from Wave failed: " + e);
            this.dataType = null;
        }
    }

    public String toString() {
        return "Wave: startTime=" + this.startTime + ", endTime=" + this.getEndTime() + ", samplingRate=" + this.samplingRate + ", samples=" + this.buffer.length + "\nstartDate=" + Util.j2KToDateString(this.startTime) + "\nendDate=" + Util.j2KToDateString(this.getEndTime()) + (this.dataType == null ? "" : "\ndataType=" + this.dataType);
    }

    @Override
    public int compareTo(Wave o) {
        return (int)Math.round(this.getStartTime() - o.getStartTime());
    }

    public Wave clone() {
        Wave w = new Wave(this);
        return w;
    }

    public void detrend() {
        double xm = this.buffer.length / 2;
        double ym = this.mean();
        double ssxx = 0.0;
        double ssxy = 0.0;
        for (int i = 0; i < this.buffer.length; ++i) {
            if (this.buffer[i] == NO_DATA) continue;
            ssxy += ((double)i - xm) * ((double)this.buffer[i] - ym);
            ssxx += ((double)i - xm) * ((double)i - xm);
        }
        double m = ssxy / ssxx;
        double b = ym - m * xm;
        for (int i = 0; i < this.buffer.length; ++i) {
            if (this.buffer[i] == NO_DATA) continue;
            int n = i;
            this.buffer[n] = (int)((double)this.buffer[n] - ((double)this.buffer[i] * m + b));
        }
    }

    public void despike(double p) {
        this.set2mean(p);
    }

    public void set2mean(double p) {
        int j = 0;
        double jtime = 0.0;
        Meaner window = new Meaner();
        window.add(this.buffer[0]);
        int r = this.buffer.length;
        for (int i = 1; i < r; ++i) {
            double itime = (double)i / this.samplingRate;
            double ival = this.buffer[i];
            window.add(ival);
            while (itime - jtime > p) {
                window.removeOldest();
                jtime = (double)(++j) / this.samplingRate;
            }
            this.buffer[i] = (int)Math.round(window.avg());
        }
    }

    public void set2median(double p) {
        int j = 0;
        double jtime = 0.0;
        Medianer window = new Medianer();
        window.add(this.buffer[0]);
        int r = this.buffer.length;
        for (int i = 1; i < r; ++i) {
            double itime = (double)i / this.samplingRate;
            double ival = this.buffer[i];
            window.add(ival);
            while (itime - jtime > p) {
                window.removeOldest();
                jtime = (double)(++j) / this.samplingRate;
            }
            this.buffer[i] = (int)Math.round(window.avg());
        }
    }

    private class Medianer {
        private MultiMap loHalf = new MultiMap();
        private MultiMap hiHalf = new MultiMap();
        private int idx1 = 0;
        private int idx2 = -1;

        Medianer() {
        }

        public void add(double val) {
            ++this.idx2;
            if (this.loHalf.size() == this.hiHalf.size()) {
                this.loHalf.addLo(val, this.idx2, this.hiHalf);
            } else {
                this.hiHalf.addHi(val, this.idx2, this.loHalf);
            }
        }

        public void dump() {
            System.out.print("[");
            this.loHalf.dump();
            System.out.print("]:[");
            this.hiHalf.dump();
            System.out.println("]");
        }

        public void removeOldest() {
            int hiSize;
            if (!this.loHalf.removeIndex(this.idx1)) {
                this.hiHalf.removeIndex(this.idx1);
            }
            ++this.idx1;
            int loSize = this.loHalf.size();
            if (loSize < (hiSize = this.hiHalf.size())) {
                Double d = this.hiHalf.firstKey();
                Integer ix = this.hiHalf.removeFirstIndex(d);
                this.loHalf.put(d, ix);
            } else if (loSize > hiSize + 1) {
                Double d = this.loHalf.lastKey();
                Integer ix = this.loHalf.removeLastIndex(d);
                this.hiHalf.put(d, ix);
            }
        }

        public double avg() {
            if (this.loHalf.size() == this.hiHalf.size()) {
                return (this.loHalf.lastKey() + this.hiHalf.firstKey()) / 2.0;
            }
            return this.loHalf.lastKey();
        }

        private class MultiMap {
            private TreeMap<Double, LinkedList<Integer>> mm = new TreeMap();
            private TreeMap<Integer, Double> unmm = new TreeMap();

            MultiMap() {
            }

            protected void put(Double key, Integer val) {
                LinkedList<Integer> entry = this.mm.get(key);
                if (entry == null) {
                    entry = new LinkedList();
                }
                this.unmm.put(val, key);
                if (entry.size() == 0) {
                    entry.add(val);
                    this.mm.put(key, entry);
                    return;
                }
                if (val < entry.getFirst()) {
                    entry.addFirst(val);
                } else {
                    entry.addLast(val);
                }
            }

            protected Double lastKey() {
                return this.mm.lastKey();
            }

            protected Integer removeLastIndex(Double key) {
                LinkedList<Integer> entry = this.mm.get(key);
                Integer index = entry.removeLast();
                if (entry.size() == 0) {
                    this.mm.remove(key);
                }
                this.unmm.remove(index);
                return index;
            }

            protected Double firstKey() {
                return this.mm.firstKey();
            }

            protected Integer removeFirstIndex(Double key) {
                LinkedList<Integer> entry = this.mm.get(key);
                Integer index = entry.removeFirst();
                if (entry.size() == 0) {
                    this.mm.remove(key);
                }
                this.unmm.remove(index);
                return index;
            }

            public int size() {
                return this.unmm.size();
            }

            public void addLo(Double key, Integer val, MultiMap other) {
                if (other.size() > 0 && key >= other.firstKey()) {
                    other.put(key, val);
                    key = other.firstKey();
                    val = other.removeFirstIndex(key);
                }
                this.put(key, val);
            }

            public void addHi(Double key, Integer val, MultiMap other) {
                if (other.size() > 0 && key <= other.lastKey()) {
                    other.put(key, val);
                    key = other.lastKey();
                    val = other.removeLastIndex(key);
                }
                this.put(key, val);
            }

            public boolean indexIsMember(Integer val) {
                return this.unmm.containsKey(val);
            }

            public boolean removeIndex(Integer val) {
                Double key = this.unmm.remove(val);
                if (key == null) {
                    return false;
                }
                List entry = this.mm.get(key);
                if (entry.size() == 1) {
                    this.mm.remove(key);
                } else {
                    entry.remove(val);
                }
                return true;
            }

            public void dump() {
                for (Double d : this.mm.keySet()) {
                    List d_ids = this.mm.get(d);
                    if (d_ids == null || d_ids.size() == 0) continue;
                    System.out.print(d);
                    if (d_ids.size() > 1) {
                        System.out.print("x" + this.mm.get(d).size());
                    }
                    System.out.print(" ");
                }
            }
        }
    }

    private class Meaner {
        private LinkedList<Double> data = new LinkedList();
        private double sum = 0.0;

        Meaner() {
        }

        public void add(double val) {
            this.data.addLast(val);
            this.sum += val;
        }

        public double removeOldest() {
            Double datum = this.data.removeFirst();
            this.sum -= datum.doubleValue();
            return datum;
        }

        public double avg() {
            return this.sum / (double)this.data.size();
        }
    }
}

