/*
 * Decompiled with CFR 0.152.
 */
package gov.usgs.vdx.data.wave;

import gov.usgs.math.Butterworth;
import gov.usgs.math.FFT;
import gov.usgs.math.Filter;
import gov.usgs.util.IntVector;
import gov.usgs.util.Util;
import gov.usgs.vdx.data.BinaryDataSet;
import gov.usgs.vdx.data.wave.SAC;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;
import java.util.TimeZone;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class Wave
implements BinaryDataSet,
Comparable<Wave>,
Cloneable {
    public static int NO_DATA = Integer.MIN_VALUE;
    public int[] buffer;
    private double startTime;
    private double samplingRate;
    private double registrationOffset = Double.NaN;
    private transient double mean = Double.NaN;
    private transient double rsam = Double.NaN;
    private transient int max = Integer.MIN_VALUE;
    private transient int min = Integer.MAX_VALUE;
    private transient int[] dataRange = null;

    public Wave() {
    }

    public Wave(Wave wave) {
        this.startTime = wave.startTime;
        this.samplingRate = wave.samplingRate;
        this.buffer = new int[wave.buffer.length];
        System.arraycopy(wave.buffer, 0, this.buffer, 0, wave.buffer.length);
    }

    public Wave(int[] b, double st, double sr) {
        this.buffer = b;
        this.startTime = st;
        this.samplingRate = sr;
    }

    public Wave(ByteBuffer bb) {
        this.fromBinary(bb);
    }

    public void register() {
        double n = this.startTime;
        double m = 1.0 / this.samplingRate;
        double dif = n % m;
        this.registrationOffset = dif >= m / 2.0 ? m - dif : -dif;
        this.startTime += this.registrationOffset;
    }

    public void setRegistrationOffset(double o) {
        this.registrationOffset = o;
    }

    public boolean isData() {
        for (int i = 0; i < this.buffer.length; ++i) {
            if (this.buffer[i] == NO_DATA) continue;
            return true;
        }
        return false;
    }

    public double[][] toSpectrogram(int sampleSize, boolean logPower, boolean logFreq, double overlap) {
        int overlapSize = (int)((double)sampleSize * overlap);
        if (overlapSize >= sampleSize) {
            overlapSize = sampleSize - 1;
        }
        int xSize = this.buffer.length / (sampleSize - overlapSize);
        int ySize = sampleSize / 2;
        double[][] powerBuf = new double[xSize][ySize];
        double[][] smallBuf = new double[sampleSize][2];
        int m = (int)Math.round(this.mean());
        for (int i = 0; i < xSize; ++i) {
            int j;
            int si = i * (sampleSize - overlapSize);
            for (j = 0; j < sampleSize; ++j) {
                smallBuf[j][0] = si + j < this.buffer.length && this.buffer[si + j] != NO_DATA ? (double)this.buffer[si + j] : (double)m;
                smallBuf[j][1] = 0.0;
            }
            FFT.fft(smallBuf);
            for (j = 1; j < sampleSize / 2; ++j) {
                double re = smallBuf[j][0];
                double im = smallBuf[j][1];
                double mag = Math.sqrt(re * re + im * im);
                if (logPower) {
                    mag = Math.log(mag) / FFT.LOG10;
                }
                powerBuf[i][j] = mag;
            }
        }
        return powerBuf;
    }

    public void trunc(int s) {
        int[] buf = new int[s];
        System.arraycopy(this.buffer, 0, buf, 0, s);
        this.buffer = buf;
    }

    public void subtract(int m) {
        int i = 0;
        while (i < this.buffer.length) {
            int n = i++;
            this.buffer[n] = this.buffer[n] - m;
        }
    }

    public void setSamplingRate(double sr) {
        this.samplingRate = sr;
    }

    public void setStartTime(double st) {
        this.startTime = st;
    }

    public double[][] fft() {
        int i;
        int n = this.buffer.length;
        int p2 = (int)Math.ceil(Math.log(n) / Math.log(2.0));
        int newSize = (int)Math.pow(2.0, p2);
        double[][] buf = new double[newSize][2];
        int m = (int)Math.round(this.mean());
        for (i = 0; i < n; ++i) {
            if (this.buffer[i] == NO_DATA) continue;
            buf[i][0] = this.buffer[i];
        }
        for (i = n; i < newSize; ++i) {
            buf[i][0] = m;
        }
        FFT.fft(buf);
        return buf;
    }

    public void invalidateStatistics() {
        this.mean = Double.NaN;
        this.rsam = Double.NaN;
        this.max = Integer.MIN_VALUE;
        this.min = Integer.MAX_VALUE;
    }

    private void deriveStatistics() {
        if (this.buffer == null || this.buffer.length == 0) {
            this.mean = 0.0;
            this.rsam = 0.0;
            this.max = 0;
            this.min = 0;
            return;
        }
        int noDatas = 0;
        long sum = 0L;
        long rs = 0L;
        for (int i = 0; i < this.buffer.length; ++i) {
            int d = this.buffer[i];
            if (d != NO_DATA) {
                sum += (long)d;
                rs += (long)Math.abs(d);
                this.min = Math.min(this.min, d);
                this.max = Math.max(this.max, d);
                continue;
            }
            ++noDatas;
        }
        this.mean = (double)sum / (double)(this.buffer.length - noDatas);
        this.rsam = (double)rs / (double)(this.buffer.length - noDatas);
        this.dataRange = new int[]{this.min, this.max};
    }

    public double mean() {
        if (Double.isNaN(this.mean)) {
            this.deriveStatistics();
        }
        return this.mean;
    }

    public int max() {
        if (this.max == Integer.MIN_VALUE) {
            this.deriveStatistics();
        }
        return this.max;
    }

    public int min() {
        if (this.min == Integer.MAX_VALUE) {
            this.deriveStatistics();
        }
        return this.min;
    }

    public double rsam() {
        if (Double.isNaN(this.rsam)) {
            this.deriveStatistics();
        }
        return this.rsam;
    }

    public void decimate(int factor) {
        int[] buf = new int[this.samples() / factor];
        for (int i = 0; i < this.samples() / factor; ++i) {
            buf[i] = this.buffer[i * factor];
        }
        this.buffer = buf;
        this.samplingRate /= (double)factor;
    }

    public int samples() {
        return this.buffer.length;
    }

    public void convertToJ2K() {
        this.startTime = Util.ewToJ2K(this.startTime);
    }

    public double getSamplingRate() {
        return this.samplingRate;
    }

    public double getStartTime() {
        return this.startTime;
    }

    public double getEndTime() {
        return this.startTime + (double)this.samples() * 1.0 / this.samplingRate;
    }

    public double getRegistrationOffset() {
        return this.registrationOffset;
    }

    public int[] getDataRange() {
        if (this.min == Integer.MAX_VALUE) {
            this.deriveStatistics();
        }
        return this.dataRange;
    }

    public boolean adjacent(Wave wave) {
        if (this.getEndTime() == wave.getStartTime()) {
            return true;
        }
        return this.getStartTime() == wave.getEndTime();
    }

    public boolean overlaps(Wave wave) {
        if (this.getEndTime() <= wave.getStartTime()) {
            return false;
        }
        return !(this.getStartTime() >= wave.getEndTime());
    }

    public boolean overlaps(double t1, double t2) {
        if (this.getEndTime() <= t1) {
            return false;
        }
        return !(this.getStartTime() >= t2);
    }

    public Wave[] split() {
        Wave sw1 = new Wave();
        Wave sw2 = new Wave();
        sw1.startTime = this.startTime;
        sw1.samplingRate = this.samplingRate;
        int length1 = this.buffer.length / 2;
        sw1.buffer = new int[length1];
        System.arraycopy(this.buffer, 0, sw1.buffer, 0, length1);
        sw2.startTime = this.startTime + (double)length1 * (1.0 / this.samplingRate);
        sw2.samplingRate = this.samplingRate;
        int length2 = this.buffer.length / 2 + this.buffer.length % 2;
        sw2.buffer = new int[length2];
        System.arraycopy(this.buffer, length1, sw2.buffer, 0, length2);
        return new Wave[]{sw1, sw2};
    }

    public List<Wave> split(int maxSamples) {
        int numSamples;
        ArrayList<Wave> list = new ArrayList<Wave>(this.buffer.length / maxSamples + 2);
        double ct = this.startTime;
        for (int j = 0; j < this.buffer.length; j += numSamples) {
            numSamples = Math.min(maxSamples, this.buffer.length - j);
            Wave sw = new Wave();
            sw.startTime = ct;
            sw.samplingRate = this.samplingRate;
            sw.buffer = new int[numSamples];
            System.arraycopy(this.buffer, j, sw.buffer, 0, numSamples);
            ct += (double)numSamples * (1.0 / this.samplingRate);
            list.add(sw);
        }
        return list;
    }

    public Wave subset(double t1, double t2) {
        if (t1 < this.getStartTime() || t2 > this.getEndTime() || t2 < t1) {
            return this;
        }
        int samples = (int)Math.floor((t2 - t1) * this.samplingRate);
        int offset = (int)Math.floor((t1 - this.startTime) * this.samplingRate);
        Wave sw = new Wave();
        sw.startTime = t1;
        sw.samplingRate = this.samplingRate;
        sw.registrationOffset = this.registrationOffset;
        sw.buffer = new int[samples];
        System.arraycopy(this.buffer, offset, sw.buffer, 0, samples);
        return sw;
    }

    public void erase(double t1, double t2) {
        int[] buf;
        int size;
        if (t2 < this.getStartTime() || t1 > this.getEndTime()) {
            return;
        }
        if (t1 >= this.getStartTime() && t2 <= this.getEndTime()) {
            return;
        }
        if (t1 <= this.getStartTime() && t2 >= this.getEndTime()) {
            this.buffer = null;
            this.startTime = Double.NaN;
            this.samplingRate = Double.NaN;
        }
        if (t2 > this.getStartTime() && t2 <= this.getEndTime()) {
            size = (int)Math.round((t2 - this.getStartTime()) * this.samplingRate);
            buf = new int[this.buffer.length - size];
            System.arraycopy(this.buffer, size, buf, 0, this.buffer.length - size);
            this.buffer = buf;
            this.startTime = t2;
        }
        if (t1 >= this.getStartTime() && t1 <= this.getEndTime()) {
            size = (int)Math.round((this.getEndTime() - t1) * this.samplingRate);
            buf = new int[this.buffer.length - size];
            System.arraycopy(this.buffer, 0, buf, 0, this.buffer.length - size);
            this.buffer = buf;
        }
    }

    public Wave combine(Wave wave) {
        if (this.samplingRate != wave.getSamplingRate()) {
            return null;
        }
        if (this.startTime >= wave.getStartTime() && this.getEndTime() <= wave.getEndTime()) {
            return wave;
        }
        if (this.startTime <= wave.getStartTime() && this.getEndTime() >= wave.getEndTime()) {
            return this;
        }
        if (this.startTime <= wave.getStartTime()) {
            int[] newbuf = new int[(int)Math.round((wave.getEndTime() - this.startTime) * this.samplingRate)];
            System.arraycopy(this.buffer, 0, newbuf, 0, this.buffer.length);
            int len = (int)Math.round((wave.getEndTime() - this.getEndTime()) * this.samplingRate);
            int i = (int)((this.getEndTime() - wave.getStartTime()) * this.samplingRate);
            System.arraycopy(wave.buffer, i, newbuf, this.buffer.length, len);
            this.buffer = newbuf;
            return this;
        }
        if (wave.getStartTime() <= this.startTime) {
            double dt = this.getEndTime() - wave.startTime;
            int[] newbuf = new int[(int)Math.round(dt * this.samplingRate)];
            System.arraycopy(wave.buffer, 0, newbuf, 0, wave.buffer.length);
            int len = (int)Math.round((this.getEndTime() - wave.getEndTime()) * this.samplingRate);
            int i = (int)((wave.getEndTime() - this.getStartTime()) * this.samplingRate);
            System.arraycopy(this.buffer, i, newbuf, wave.buffer.length, len);
            this.buffer = newbuf;
            this.startTime = wave.startTime;
            return this;
        }
        return null;
    }

    public static Wave join(List<Wave> waves) {
        if (waves == null || waves.size() == 0) {
            return null;
        }
        if (waves.size() == 1) {
            return waves.get(0);
        }
        double mint = 1.0E300;
        double maxt = -1.0E300;
        double sr = -1.0;
        for (Wave sw : waves) {
            mint = Math.min(mint, sw.getStartTime());
            maxt = Math.max(maxt, sw.getEndTime());
            sr = sw.getSamplingRate();
        }
        int samples = (int)((maxt - mint) * sr);
        int[] buffer = new int[samples + 1];
        Arrays.fill(buffer, NO_DATA);
        for (Wave sw : waves) {
            int i = (int)Math.round((sw.getStartTime() - mint) * sr);
            System.arraycopy(sw.buffer, 0, buffer, i, sw.buffer.length);
        }
        return new Wave(buffer, mint, sr);
    }

    public static Wave join(List<Wave> waves, double t1, double t2) {
        if (waves == null || waves.size() == 0) {
            return null;
        }
        Wave wv0 = waves.get(0);
        int samples = (int)((t2 - t1) * wv0.getSamplingRate());
        int[] buffer = new int[samples + 1];
        Arrays.fill(buffer, NO_DATA);
        for (Wave sw : waves) {
            int i = (int)Math.round((sw.getStartTime() - t1) * wv0.getSamplingRate());
            System.arraycopy(sw.buffer, 0, buffer, i, sw.buffer.length);
        }
        return new Wave(buffer, t1, wv0.getSamplingRate());
    }

    public int getMemorySize() {
        if (this.buffer == null) {
            return 0;
        }
        return this.buffer.length * 4;
    }

    public void exportToText(String fn) throws IOException {
        PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fn)));
        double ct = this.startTime + 9.46728E8;
        double ts = 1.0 / this.samplingRate;
        for (int i = 0; i < this.buffer.length; ++i) {
            out.println(Math.round(ct * 1000.0) + " " + this.buffer[i]);
            ct += ts;
        }
        out.close();
    }

    public static Wave importFromText(String fn) {
        try {
            IntVector iv = new IntVector(5000, 10000);
            BufferedReader in = new BufferedReader(new FileReader(fn));
            Wave sw = new Wave();
            String firstLine = in.readLine();
            sw.startTime = Double.parseDouble(firstLine.substring(0, firstLine.indexOf(" ")));
            iv.add(Integer.parseInt(firstLine.substring(firstLine.indexOf(" ") + 1)));
            String secondLine = in.readLine();
            double nt = Double.parseDouble(secondLine.substring(0, secondLine.indexOf(" ")));
            sw.samplingRate = 1.0 / ((nt - sw.startTime) / 1000.0);
            String line = secondLine;
            do {
                iv.add(Integer.parseInt(line.substring(line.indexOf(" ") + 1)));
            } while ((line = in.readLine()) != null);
            sw.startTime /= 1000.0;
            sw.startTime -= 9.46728E8;
            sw.buffer = iv.getResizedInts();
            in.close();
            return sw;
        }
        catch (Exception exception) {
            return null;
        }
    }

    public double[] integrate() {
        double[] d = new double[this.buffer.length];
        double period = 1.0 / this.samplingRate;
        double mean = this.mean();
        double sum = 0.0;
        for (int i = 0; i < this.buffer.length; ++i) {
            d[i] = (sum += (double)this.buffer[i] - mean) * period;
        }
        return d;
    }

    public double[] removeMean() {
        double[] d = new double[this.buffer.length];
        double mean = this.mean();
        for (int i = 0; i < this.buffer.length; ++i) {
            d[i] = (double)this.buffer[i] - mean;
        }
        return d;
    }

    public void filter(Butterworth bw, boolean zeroPhaseShift) {
        int i;
        double mean = this.mean();
        double[] dBuf = new double[this.buffer.length + (int)((double)this.buffer.length * 0.5)];
        Arrays.fill(dBuf, mean);
        int trueStart = (int)((double)this.buffer.length * 0.25);
        for (i = 0; i < this.buffer.length; ++i) {
            if (this.buffer[i] == NO_DATA) continue;
            dBuf[i + trueStart] = this.buffer[i];
        }
        bw.setSamplingRate(this.getSamplingRate());
        bw.create();
        Filter.filter(dBuf, bw.getSize(), bw.getXCoeffs(), bw.getYCoeffs(), bw.getGain(), 0.0, 0.0);
        if (zeroPhaseShift) {
            double[] dBuf2 = new double[dBuf.length];
            int i2 = 0;
            int j = dBuf.length - 1;
            while (i2 < dBuf.length) {
                dBuf2[j] = dBuf[i2];
                ++i2;
                --j;
            }
            Filter.filter(dBuf2, bw.getSize(), bw.getXCoeffs(), bw.getYCoeffs(), bw.getGain(), 0.0, 0.0);
            i2 = 0;
            j = dBuf2.length - 1 - trueStart;
            while (i2 < this.buffer.length) {
                this.buffer[i2] = (int)Math.round(dBuf2[j]);
                ++i2;
                --j;
            }
        } else {
            for (i = 0; i < this.buffer.length; ++i) {
                this.buffer[i] = (int)Math.round(dBuf[i + trueStart]);
            }
        }
        this.invalidateStatistics();
    }

    @Override
    public ByteBuffer toBinary() {
        ByteBuffer bb = ByteBuffer.allocate(28 + 4 * this.buffer.length);
        bb.putDouble(this.startTime);
        bb.putDouble(this.samplingRate);
        bb.putDouble(this.registrationOffset);
        bb.putInt(this.buffer.length);
        for (int i = 0; i < this.buffer.length; ++i) {
            bb.putInt(this.buffer[i]);
        }
        return bb;
    }

    @Override
    public void fromBinary(ByteBuffer bb) {
        this.startTime = bb.getDouble();
        this.samplingRate = bb.getDouble();
        this.registrationOffset = bb.getDouble();
        int length = bb.getInt();
        this.buffer = new int[length];
        for (int i = 0; i < length; ++i) {
            this.buffer[i] = bb.getInt();
        }
    }

    public String toString() {
        return "Wave: startTime=" + this.startTime + ", endTime=" + this.getEndTime() + ", samplingRate=" + this.samplingRate + ", samples=" + this.buffer.length + "\nstartDate=" + Util.j2KToDateString(this.startTime) + "\nendDate=" + Util.j2KToDateString(this.getEndTime());
    }

    public SAC toSAC() {
        SAC sac = new SAC();
        sac.npts = this.buffer.length;
        Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        cal.setTime(Util.j2KToDate(this.startTime));
        sac.nzyear = cal.get(1);
        sac.nzjday = cal.get(6);
        sac.nzhour = cal.get(11);
        sac.nzmin = cal.get(12);
        sac.nzsec = cal.get(13);
        sac.nzmsec = cal.get(14);
        sac.delta = (float)(1.0 / this.samplingRate);
        sac.y = new float[this.buffer.length];
        for (int i = 0; i < this.buffer.length; ++i) {
            sac.y[i] = this.buffer[i];
        }
        sac.b = 0.0f;
        sac.e = (float)this.buffer.length * sac.delta;
        sac.fmt = 2.0f;
        sac.iftype = 1;
        sac.idep = 5;
        sac.iztype = 9;
        sac.leven = 1;
        sac.lpspol = 1;
        sac.lovrok = 1;
        sac.lcalda = 1;
        sac.kevnm = SAC.STRING8_UNDEF;
        sac.ko = "origin";
        return sac;
    }

    @Override
    public int compareTo(Wave o) {
        return (int)Math.round(this.getStartTime() - o.getStartTime());
    }

    public Wave clone() {
        Wave w = new Wave(this);
        return w;
    }
}

