/*
 * Decompiled with CFR 0.152.
 */
package gov.usgs.vdx.db;

import gov.usgs.util.Arguments;
import gov.usgs.util.ConfigFile;
import gov.usgs.util.Log;
import gov.usgs.util.Retriable;
import gov.usgs.util.Util;
import gov.usgs.vdx.data.SQLDataSource;
import gov.usgs.vdx.data.generic.fixed.SQLGenericFixedDataSource;
import gov.usgs.vdx.data.generic.variable.SQLGenericVariableDataSource;
import gov.usgs.vdx.data.gps.SQLGPSDataSource;
import gov.usgs.vdx.data.hypo.SQLHypocenterDataSource;
import gov.usgs.vdx.data.rsam.SQLEWRSAMDataSource;
import gov.usgs.vdx.data.rsam.SQLRSAMDataSource;
import gov.usgs.vdx.data.tilt.SQLTiltDataSource;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

public class VDXDatabase {
    private static final String DEFAULT_CONFIG_FILE = "VDX.config";
    private static final String CURRENT_SCHEMA_VERSION = "1.0.0";
    private static final String DEFAULT_DATABASE_PREFIX = "V";
    private Connection connection;
    private Statement statement;
    private boolean connected;
    private String dbDriver;
    private String dbURL;
    private String dbPrefix = "V";
    private Logger logger = Log.getLogger("gov.usgs.vdx");
    private Map<String, PreparedStatement> preparedStatements;

    public VDXDatabase(String driver, String url, String prefix) {
        this.logger.finest("New VDXDatabase: " + driver + ":" + url + ":" + prefix);
        this.dbDriver = driver;
        this.dbURL = url;
        if (prefix != null) {
            this.dbPrefix = prefix;
        }
        this.preparedStatements = new HashMap<String, PreparedStatement>();
        this.connect();
    }

    public static VDXDatabase getVDXDatabase(String cf) {
        VDXDatabase db = null;
        try {
            ConfigFile config = new ConfigFile(cf);
            String driver = config.getString("vdx.driver");
            String url = config.getString("vdx.url");
            String prefix = config.getString("vdx.prefix");
            if (driver == null) {
                throw new RuntimeException("Can't find config parameter vdx.driver.");
            }
            if (url == null) {
                throw new RuntimeException("Can't find config parameter vdx.url.");
            }
            if (prefix == null) {
                throw new RuntimeException("Can't find config parameter vdx.prefix.");
            }
            db = new VDXDatabase(driver, url, prefix);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        return db;
    }

    public Logger getLogger() {
        return this.logger;
    }

    public void setLogger(Logger log) {
        this.logger = log;
    }

    public void connect() {
        this.logger.fine("Connecting to " + this.dbURL);
        this.connected = false;
        try {
            Class.forName(this.dbDriver).newInstance();
            DriverManager.setLoginTimeout(3);
            this.connection = DriverManager.getConnection(this.dbURL);
            this.statement = this.connection.createStatement();
            this.connected = true;
            this.preparedStatements.clear();
        }
        catch (ClassNotFoundException e) {
            this.logger.log(Level.SEVERE, "Could not load the database driver, check your CLASSPATH.", Util.getLineNumber(this, e));
            System.exit(-1);
        }
        catch (Exception e) {
            this.connection = null;
            this.statement = null;
            this.logger.log(Level.SEVERE, "Could not connect to VDX.", e);
            this.connected = false;
        }
    }

    public void close() {
        if (!this.checkConnect()) {
            return;
        }
        try {
            this.statement.close();
            this.connection.close();
            this.connected = false;
        }
        catch (Exception e) {
            this.logger.warning("Error closing database.  This is unusual, but not critical.");
        }
    }

    public boolean checkConnect() {
        return this.checkConnect(true);
    }

    public boolean checkConnect(boolean verbose) {
        if (this.connected) {
            return true;
        }
        new Retriable<Object>(){

            @Override
            public boolean attempt() {
                VDXDatabase.this.connect();
                return VDXDatabase.this.connected;
            }
        }.go();
        return this.connected;
    }

    public boolean connected() {
        return this.connected;
    }

    public Connection getConnection() {
        return this.connection;
    }

    public Statement getStatement() {
        return this.statement;
    }

    public boolean execute(final String sql) {
        Boolean b = (Boolean)new Retriable<Boolean>(){

            @Override
            public void attemptFix() {
                VDXDatabase.this.close();
                VDXDatabase.this.connect();
            }

            @Override
            public boolean attempt() {
                try {
                    VDXDatabase.this.statement.execute(sql);
                    this.result = new Boolean(true);
                    return true;
                }
                catch (SQLException e) {
                    VDXDatabase.this.logger.log(Level.SEVERE, "execute() failed, SQL: " + sql, e);
                    this.result = new Boolean(false);
                    return false;
                }
            }
        }.go();
        return b != null && b != false;
    }

    public ResultSet executeQuery(final String sql) {
        ResultSet rs = (ResultSet)new Retriable<ResultSet>(){

            @Override
            public void attemptFix() {
                VDXDatabase.this.close();
                VDXDatabase.this.connect();
            }

            @Override
            public boolean attempt() {
                try {
                    this.result = VDXDatabase.this.statement.executeQuery(sql);
                    return true;
                }
                catch (SQLException e) {
                    VDXDatabase.this.logger.log(Level.SEVERE, "executeQuery() failed, SQL: " + sql, e);
                    return false;
                }
            }
        }.go();
        return rs;
    }

    public String getDatabasePrefix() {
        return this.dbPrefix;
    }

    private void createTables() {
        try {
            this.useRootDatabase();
            this.getStatement().execute("CREATE TABLE version (schemaversion VARCHAR(10), installtime DATETIME)");
            this.getStatement().execute("INSERT INTO version VALUES ('1.0.0', NOW())");
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Could not create table in VDX database.  Are permissions set properly?", Util.getLineNumber(this, e));
        }
    }

    public boolean useRootDatabase() {
        return this.useDatabase("ROOT");
    }

    public boolean useDatabase(String db) {
        db = this.dbPrefix + "_" + db;
        if (!this.checkConnect()) {
            return false;
        }
        try {
            this.statement.execute("USE " + db);
            return true;
        }
        catch (SQLException e) {
            if (e.getMessage().indexOf("Unknown database") != -1) {
                this.logger.log(Level.SEVERE, db + " database does not exist");
            } else {
                this.logger.log(Level.SEVERE, db + " database connection failed", e);
            }
            return false;
        }
    }

    public boolean useV2Database(String db) {
        if (!this.checkConnect()) {
            return false;
        }
        try {
            try {
                this.statement.execute("USE " + db);
            }
            catch (SQLException e) {
                this.logger.log(Level.SEVERE, "Lost connection to VALVE 2 database, attempting to reconnect.");
                this.close();
                this.connect();
            }
            this.statement.execute("USE " + db);
            return true;
        }
        catch (SQLException e) {
            if (e.getMessage().indexOf("Unknown database") != -1) {
                this.logger.log(Level.SEVERE, "Attempt to use nonexistent database: " + db);
            } else {
                this.logger.log(Level.SEVERE, "Could not use database: " + db, e);
            }
            return false;
        }
    }

    public boolean checkDatabase() {
        if (!this.checkConnect()) {
            return false;
        }
        try {
            boolean failed = false;
            try {
                this.getStatement().execute("USE " + this.getDatabasePrefix() + "_ROOT");
            }
            catch (Exception e) {
                failed = true;
            }
            if (failed) {
                this.getStatement().execute("CREATE DATABASE " + this.getDatabasePrefix() + "_ROOT");
                this.getStatement().execute("USE " + this.getDatabasePrefix() + "_ROOT");
                this.createTables();
            }
            return true;
        }
        catch (Exception e) {
            this.logger.severe("Could not locate or create VDX database.  Are permissions set properly?");
            return false;
        }
    }

    public PreparedStatement getPreparedStatement(String sql) {
        try {
            PreparedStatement ps = this.preparedStatements.get(sql);
            if (ps == null) {
                ps = this.connection.prepareStatement(sql);
                this.preparedStatements.put(sql, ps);
            }
            return ps;
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Could not prepare statement.", e);
            return null;
        }
    }

    protected static void createDatabase(ConfigFile params, Arguments args, SQLDataSource ds) {
        String name = args.get("-n");
        if (name == null) {
            System.err.println("You must specify the name of the database with '-n'.");
            System.exit(-1);
        }
        params.put("vdx.name", name);
        ds.initialize(params);
    }

    public boolean tableExists(String db, String table) {
        try {
            ResultSet rs = this.getStatement().executeQuery(String.format("SELECT COUNT(*) FROM %s_%s.%s", this.dbPrefix, db, table));
            boolean result = rs.next();
            rs.close();
            return result;
        }
        catch (Exception exception) {
            return false;
        }
    }

    public static void main(String[] as) {
        String action;
        HashSet<String> flags = new HashSet<String>();
        HashSet<String> kvs = new HashSet<String>();
        kvs.add("-c");
        kvs.add("-n");
        kvs.add("-a");
        Arguments args = new Arguments(as, flags, kvs);
        String cf = args.get("-c");
        if (cf == null) {
            cf = DEFAULT_CONFIG_FILE;
        }
        ConfigFile params = new ConfigFile(cf);
        VDXDatabase db = VDXDatabase.getVDXDatabase(cf);
        if (db == null) {
            System.out.println("Could not connect to VDX database");
            System.exit(-1);
        }
        if ((action = args.get("-a")) == null) {
            System.out.println("<VDXDatabase> [-c configfile] -a <action> [other args]");
            System.out.println("Known actions:");
            System.out.println("creategenericfixed");
            System.out.println("creategenericvariable");
            System.out.println("creategps");
            System.out.println("createhypocenters");
            System.out.println("createewrsam");
            System.out.println("createtilt");
        } else if ((action = action.toLowerCase()).equals("createvdx")) {
            db.checkDatabase();
        } else {
            HashMap<String, SQLDataSource> sources = new HashMap<String, SQLDataSource>();
            sources.put("creategenericfixed", new SQLGenericFixedDataSource());
            sources.put("creategenericvariable", new SQLGenericVariableDataSource());
            sources.put("creategps", new SQLGPSDataSource());
            sources.put("createhypocenters", new SQLHypocenterDataSource());
            sources.put("creatersam", new SQLRSAMDataSource());
            sources.put("createewrsam", new SQLEWRSAMDataSource());
            sources.put("createtilt", new SQLTiltDataSource());
            SQLDataSource sds = (SQLDataSource)sources.get(action);
            if (sds != null) {
                VDXDatabase.createDatabase(params, args, sds);
            } else {
                System.out.println("I don't know how to " + action);
            }
        }
    }
}

